const core = require('@sentry/core');
const electron = require('electron');
const context = require('../context.js');
const headerInjection = require('../header-injection.js');
const normalize = require('../normalize.js');

// A cache of renderer profiles which need attaching to events
let RENDERER_PROFILES;
/**
 * Caches a profile to later be re-attached to an event
 */
function rendererProfileFromIpc(event, profile) {
    if (!RENDERER_PROFILES) {
        return;
    }
    const profile_id = profile.event_id;
    RENDERER_PROFILES.set(profile_id, profile);
    if (event) {
        event.contexts = {
            ...event.contexts,
            // Re-add the profile context which we can later use to find the correct profile
            profile: {
                profile_id,
            },
        };
    }
}
/**
 * Injects 'js-profiling' document policy headers and ensures that profiles get forwarded with transactions
 */
const rendererProfilingIntegration = core.defineIntegration(() => {
    return {
        name: 'RendererProfiling',
        setup(client) {
            const options = client.getOptions();
            if (!options.enableRendererProfiling) {
                return;
            }
            RENDERER_PROFILES = new core.LRUMap(10);
            electron.app.on('ready', () => {
                // Ensure the correct headers are set to enable the browser profiler
                options.getSessions().forEach((sesh) => headerInjection.addHeaderToSession(sesh, 'Document-Policy', 'js-profiling'));
            });
            // Copy the profiles back into the event envelopes
            client.on?.('beforeEnvelope', (envelope) => {
                let profile_id;
                core.forEachEnvelopeItem(envelope, (item, type) => {
                    if (type !== 'transaction') {
                        return;
                    }
                    for (let j = 1; j < item.length; j++) {
                        const event = item[j];
                        if (event?.contexts?.profile?.profile_id) {
                            profile_id = event.contexts.profile.profile_id;
                            // This can be removed as it's no longer needed
                            delete event.contexts.profile;
                        }
                    }
                });
                if (!profile_id) {
                    return;
                }
                const profile = RENDERER_PROFILES?.remove(profile_id);
                if (!profile) {
                    return;
                }
                normalize.normaliseProfile(profile, electron.app.getAppPath());
                profile.release = options.release || context.getDefaultReleaseName();
                profile.environment = options.environment || context.getDefaultEnvironment();
                // @ts-expect-error untyped envelope
                envelope[1].push([{ type: 'profile' }, profile]);
            });
        },
    };
});

exports.rendererProfileFromIpc = rendererProfileFromIpc;
exports.rendererProfilingIntegration = rendererProfilingIntegration;//# sourceMappingURL=http://go/sourcemap/sourcemaps/d44e0b8c64929a5174b4bc2b1cb43f1440bac1c0/node_modules/@sentry/electron/main/integrations/renderer-profiling.js.map
