import { debug } from '@sentry/core';
import { app } from 'electron';
import { promises } from 'fs';
import { join, basename } from 'path';
import { Mutex } from '../../mutex.js';
import { parseMinidump } from './minidump-parser.js';

/** Maximum number of days to keep a minidump before deleting it. */
const MAX_AGE_DAYS = 30;
const MS_PER_DAY = 24 * 3600 * 1000;
/** Minimum number of milliseconds a minidump should not be modified for before we assume writing is complete */
const NOT_MODIFIED_MS = 1000;
const MAX_RETRY_MS = 5000;
const RETRY_DELAY_MS = 500;
const MAX_RETRIES = MAX_RETRY_MS / RETRY_DELAY_MS;
function delay(ms) {
    return new Promise((resolve) => setTimeout(resolve, ms));
}
/**
 * Creates a minidump loader
 * @param getMinidumpPaths A function that returns paths to minidumps
 * @param preProcessFile A function that pre-processes the minidump file
 * @returns A function to fetch minidumps
 */
function createMinidumpLoader(getMinidumpPaths) {
    // The mutex protects against a whole host of reentrancy issues and race conditions.
    const mutex = new Mutex();
    return async (deleteAll, callback) => {
        // any calls to this function will be queued and run exclusively
        await mutex.runExclusive(async () => {
            for (const path of await getMinidumpPaths()) {
                try {
                    if (deleteAll) {
                        continue;
                    }
                    debug.log('Found minidump', path);
                    let stats = await promises.stat(path);
                    const thirtyDaysAgo = new Date().getTime() - MAX_AGE_DAYS * MS_PER_DAY;
                    if (stats.mtimeMs < thirtyDaysAgo) {
                        debug.log(`Ignoring minidump as it is over ${MAX_AGE_DAYS} days old`);
                        continue;
                    }
                    let retries = 0;
                    while (retries <= MAX_RETRIES) {
                        const twoSecondsAgo = new Date().getTime() - NOT_MODIFIED_MS;
                        if (stats.mtimeMs < twoSecondsAgo) {
                            const data = await promises.readFile(path);
                            try {
                                const parsedMinidump = parseMinidump(data);
                                debug.log('Sending minidump');
                                await callback(parsedMinidump, {
                                    attachmentType: 'event.minidump',
                                    filename: basename(path),
                                    data,
                                });
                            }
                            catch (e) {
                                const message = e instanceof Error ? e.toString() : 'Unknown error';
                                debug.warn(`Dropping minidump:\n${message}`);
                                break;
                            }
                            break;
                        }
                        debug.log(`Waiting. Minidump has been modified in the last ${NOT_MODIFIED_MS} milliseconds.`);
                        retries += 1;
                        await delay(RETRY_DELAY_MS);
                        // update the stats
                        stats = await promises.stat(path);
                    }
                    if (retries >= MAX_RETRIES) {
                        debug.warn('Timed out waiting for minidump to stop being modified');
                    }
                }
                catch (e) {
                    debug.error('Failed to load minidump', e);
                }
                finally {
                    // We always attempt to delete the minidump
                    try {
                        await promises.unlink(path);
                    }
                    catch (e) {
                        debug.warn('Could not delete minidump', path);
                    }
                }
            }
        });
    };
}
/** Attempts to remove the metadata file so Crashpad doesn't output `failed to stat report` errors to the console */
async function deleteCrashpadMetadataFile(crashesDirectory, waitMs = 100) {
    if (waitMs > 2000) {
        return;
    }
    const metadataPath = join(crashesDirectory, 'metadata');
    try {
        await promises.unlink(metadataPath);
        debug.log('Deleted Crashpad metadata file', metadataPath);
    }
    catch (e) {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
        if (e.code && e.code == 'EBUSY') {
            // Since Crashpad probably still has the metadata file open, we make a few attempts to delete it, backing
            // off and waiting longer each time.
            setTimeout(async () => {
                await deleteCrashpadMetadataFile(crashesDirectory, waitMs * 2);
            }, waitMs);
        }
    }
}
async function readDirsAsync(paths) {
    const found = [];
    for (const path of paths) {
        try {
            const files = await promises.readdir(path);
            found.push(...files.map((file) => join(path, file)));
        }
        catch (_) {
            //
        }
    }
    return found;
}
/**
 * Gets the crashpad minidump loader
 */
function getMinidumpLoader() {
    const crashesDirectory = app.getPath('crashDumps');
    const crashpadSubDirectory = process.platform === 'win32' ? 'reports' : 'completed';
    const dumpDirectories = [join(crashesDirectory, crashpadSubDirectory)];
    if (process.platform === 'darwin') {
        dumpDirectories.push(join(crashesDirectory, 'pending'));
    }
    return createMinidumpLoader(async () => {
        await deleteCrashpadMetadataFile(crashesDirectory).catch((error) => debug.error(error));
        const files = await readDirsAsync(dumpDirectories);
        return files.filter((file) => file.endsWith('.dmp'));
    });
}

export { createMinidumpLoader, getMinidumpLoader };//# sourceMappingURL=http://go/sourcemap/sourcemaps/d44e0b8c64929a5174b4bc2b1cb43f1440bac1c0/node_modules/@sentry/electron/esm/main/integrations/sentry-minidump/minidump-loader.js.map
