import { defineIntegration, captureMessage, addBreadcrumb } from '@sentry/core';
import { childProcessIntegration as childProcessIntegration$1, logger } from '@sentry/node';
import { app } from 'electron';
import { EXIT_REASONS } from '../electron-normalize.js';

const DEFAULT_OPTIONS = {
    breadcrumbs: EXIT_REASONS,
    events: ['abnormal-exit', 'launch-failed', 'integrity-failure'],
};
/** Gets message and severity */
function getMessageAndSeverity(reason, process) {
    const message = `'${process}' process exited with '${reason}'`;
    const messageFmt = logger.fmt `'${process}' process exited with '${reason}'`;
    switch (reason) {
        case 'abnormal-exit':
        case 'killed':
            return { message, level: 'warning', log: logger.warn, messageFmt };
        case 'crashed':
        case 'oom':
        case 'launch-failed':
        case 'integrity-failure':
            return { message, level: 'fatal', log: logger.error, messageFmt };
        default:
            return { message, level: 'debug', log: logger.info, messageFmt };
    }
}
/**
 * Adds breadcrumbs for:
 * - Electron child process events
 * - Node `child_process` events
 * - Node `worker_threads` events
 */
const childProcessIntegration = defineIntegration((userOptions = {}) => {
    const { breadcrumbs, events } = userOptions;
    const nodeIntegration = childProcessIntegration$1(userOptions);
    const options = {
        breadcrumbs: Array.isArray(breadcrumbs) ? breadcrumbs : breadcrumbs === false ? [] : DEFAULT_OPTIONS.breadcrumbs,
        events: Array.isArray(events) ? events : events === false ? [] : DEFAULT_OPTIONS.events,
    };
    return {
        name: 'ChildProcess',
        setup(client) {
            nodeIntegration.setup?.(client);
            const { breadcrumbs, events } = options;
            const allReasons = Array.from(new Set([...breadcrumbs, ...events]));
            // only hook these events if we're after more than just the unresponsive event
            if (allReasons.length > 0) {
                const clientOptions = client.getOptions();
                const enableLogs = !!clientOptions?.enableLogs;
                app.on('child-process-gone', (_, details) => {
                    const { reason } = details;
                    const { message, level, log, messageFmt } = getMessageAndSeverity(details.reason, details.type);
                    // Capture message first
                    if (events.includes(reason)) {
                        captureMessage(message, { level, tags: { 'event.process': details.type } });
                    }
                    // And then add breadcrumbs for subsequent events
                    if (breadcrumbs.includes(reason)) {
                        addBreadcrumb({
                            type: 'process',
                            category: 'child-process',
                            message,
                            level,
                            data: details,
                        });
                        if (enableLogs) {
                            log(messageFmt, {
                                'sentry.origin': 'auto.electron.child-process',
                                exitCode: details.exitCode,
                                name: details.name,
                                serviceName: details.serviceName,
                            });
                        }
                    }
                });
                app.on('render-process-gone', (_, contents, details) => {
                    const { reason } = details;
                    const name = clientOptions?.getRendererName?.(contents) || 'renderer';
                    const { message, level, log, messageFmt } = getMessageAndSeverity(details.reason, name);
                    // Capture message first
                    if (events.includes(reason)) {
                        captureMessage(message, level);
                    }
                    // And then add breadcrumbs for subsequent events
                    if (breadcrumbs.includes(reason)) {
                        addBreadcrumb({
                            type: 'process',
                            category: 'child-process',
                            message,
                            level,
                            data: details,
                        });
                        if (enableLogs) {
                            log(messageFmt, {
                                'sentry.origin': 'auto.electron.child-process',
                                exitCode: details.exitCode,
                            });
                        }
                    }
                });
            }
        },
    };
});

export { childProcessIntegration };//# sourceMappingURL=http://go/sourcemap/sourcemaps/cf8353edc265f5e46b798bfb276861d0bf3bf120/node_modules/@sentry/electron/esm/main/integrations/child-process.js.map
