import { withMonitor, captureException } from '@sentry/core';
import { replaceCronNames } from './common.js';

/**
 * Wraps the `node-cron` library with check-in monitoring.
 *
 * ```ts
 * import * as Sentry from "@sentry/node";
 * import * as cron from "node-cron";
 *
 * const cronWithCheckIn = Sentry.cron.instrumentNodeCron(cron);
 *
 * cronWithCheckIn.schedule(
 *   "* * * * *",
 *   () => {
 *     console.log("running a task every minute");
 *   },
 *   { name: "my-cron-job" },
 * );
 * ```
 */
function instrumentNodeCron(lib) {
  return new Proxy(lib, {
    get(target, prop) {
      if (prop === 'schedule' && target.schedule) {
        // When 'get' is called for schedule, return a proxied version of the schedule function
        return new Proxy(target.schedule, {
          apply(target, thisArg, argArray) {
            const [expression, callback, options] = argArray;

            const name = options?.name;
            const timezone = options?.timezone;

            if (!name) {
              throw new Error('Missing "name" for scheduled job. A name is required for Sentry check-in monitoring.');
            }

            const monitoredCallback = async () => {
              return withMonitor(
                name,
                async () => {
                  // We have to manually catch here and capture the exception because node-cron swallows errors
                  // https://github.com/node-cron/node-cron/issues/399
                  try {
                    return await callback();
                  } catch (e) {
                    captureException(e, {
                      mechanism: {
                        handled: false,
                        type: 'auto.function.node-cron.instrumentNodeCron',
                      },
                    });
                    throw e;
                  }
                },
                {
                  schedule: { type: 'crontab', value: replaceCronNames(expression) },
                  timezone,
                },
              );
            };

            return target.apply(thisArg, [expression, monitoredCallback, options]);
          },
        });
      } else {
        return target[prop ];
      }
    },
  });
}

export { instrumentNodeCron };//# sourceMappingURL=http://go/sourcemap/sourcemaps/c68681c13bdb0dea7fb85526bcc3202d20233fa0/node_modules/@sentry/node-core/build/esm/cron/node-cron.js.map
