Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const exports$1 = require('../exports.js');
const debugLogger = require('./debug-logger.js');
const vercelWaitUntil = require('./vercelWaitUntil.js');
const worldwide = require('./worldwide.js');

async function flushWithTimeout(timeout) {
  try {
    debugLogger.debug.log('Flushing events...');
    await exports$1.flush(timeout);
    debugLogger.debug.log('Done flushing events');
  } catch (e) {
    debugLogger.debug.log('Error while flushing events:\n', e);
  }
}

/**
 *  Flushes the event queue with a timeout in serverless environments to ensure that events are sent to Sentry before the
 *  serverless function execution ends.
 *
 * The function is async, but in environments that support a `waitUntil` mechanism, it will run synchronously.
 *
 * This function is aware of the following serverless platforms:
 * - Cloudflare: If a Cloudflare context is provided, it will use `ctx.waitUntil()` to flush events (keeps the `this` context of `ctx`).
 *               If a `cloudflareWaitUntil` function is provided, it will use that to flush events (looses the `this` context of `ctx`).
 * - Vercel: It detects the Vercel environment and uses Vercel's `waitUntil` function.
 * - Other Serverless (AWS Lambda, Google Cloud, etc.): It detects the environment via environment variables
 *   and uses a regular `await flush()`.
 *
 *  @internal This function is supposed for internal Sentry SDK usage only.
 *  @hidden
 */
async function flushIfServerless(
  params

 = {},
) {
  const { timeout = 2000 } = params;

  if ('cloudflareWaitUntil' in params && typeof params?.cloudflareWaitUntil === 'function') {
    params.cloudflareWaitUntil(flushWithTimeout(timeout));
    return;
  }

  if ('cloudflareCtx' in params && typeof params.cloudflareCtx?.waitUntil === 'function') {
    params.cloudflareCtx.waitUntil(flushWithTimeout(timeout));
    return;
  }

  // @ts-expect-error This is not typed
  if (worldwide.GLOBAL_OBJ[Symbol.for('@vercel/request-context')]) {
    // Vercel has a waitUntil equivalent that works without execution context
    vercelWaitUntil.vercelWaitUntil(flushWithTimeout(timeout));
    return;
  }

  if (typeof process === 'undefined') {
    return;
  }

  const isServerless =
    !!process.env.FUNCTIONS_WORKER_RUNTIME || // Azure Functions
    !!process.env.LAMBDA_TASK_ROOT || // AWS Lambda
    !!process.env.K_SERVICE || // Google Cloud Run
    !!process.env.CF_PAGES || // Cloudflare Pages
    !!process.env.VERCEL ||
    !!process.env.NETLIFY;

  if (isServerless) {
    // Use regular flush for environments without a generic waitUntil mechanism
    await flushWithTimeout(timeout);
  }
}

exports.flushIfServerless = flushIfServerless;//# sourceMappingURL=http://go/sourcemap/sourcemaps/c25eb90df95d64f6d280779237c1ca39f9f3eef0/node_modules/@sentry/core/build/cjs/utils/flushIfServerless.js.map
