const core = require('@sentry/core');
const electron = require('electron');
const scope = require('../../../common/scope.js');
const context = require('../../context.js');
const electronNormalize = require('../../electron-normalize.js');
const renderers = require('../../renderers.js');
const sessions = require('../../sessions.js');
const store = require('../../store.js');
const minidumpLoader = require('./minidump-loader.js');

/**
 * Sends minidumps via the Sentry uploader
 */
const sentryMinidumpIntegration = core.defineIntegration((options = {}) => {
    // The remaining number of minidumps that can be sent in this session
    let minidumpsRemaining = options.maxMinidumpsPerSession || 10;
    // Store to persist context information beyond application crashes.
    let scopeStore;
    // We need to store the scope in a variable here so it can be attached to minidumps
    let scopeLastRun;
    let minidumpLoader$1;
    function startCrashReporter() {
        core.debug.log('Starting Electron crashReporter');
        electron.crashReporter.start({
            companyName: '',
            ignoreSystemCrashHandler: true,
            productName: electron.app.name || electron.app.getName(),
            // Empty string doesn't work for Linux Crashpad and no submitURL doesn't work for older versions of Electron
            submitURL: 'https://f.a.k/e',
            uploadToServer: false,
            compress: true,
        });
    }
    function setupScopeListener(client) {
        function scopeChanged(scope) {
            // Since the initial scope read is async, we need to ensure that any writes do not beat that
            // https://github.com/getsentry/sentry-electron/issues/585
            setImmediate(async () => scopeStore?.set({
                scope,
                event: await context.getEventDefaults(client),
            }));
        }
        scope.addScopeListener((scope) => {
            scopeChanged(scope);
        });
        scopeChanged(scope.getScopeData());
    }
    async function sendNativeCrashes(client, getEvent) {
        // Whenever we are called, assume that the crashes we are going to load down
        // below have occurred recently. This means, we can use the same event data
        // for all minidumps that we load now. There are two conditions:
        //
        //  1. The application crashed and we are just starting up. The stored
        //     breadcrumbs and context reflect the state during the application
        //     crash.
        //
        //  2. A renderer process crashed recently and we have just been notified
        //     about it. Just use the breadcrumbs and context information we have
        //     right now and hope that the delay was not too long.
        if (minidumpsRemaining <= 0) {
            core.debug.log('Not sending minidumps because the limit has been reached');
        }
        // If the SDK is not enabled, or we've already reached the minidump limit, tell the loader to delete all minidumps
        const deleteAll = client.getOptions().enabled === false || minidumpsRemaining <= 0;
        let minidumpFound = false;
        await minidumpLoader$1?.(deleteAll, async (minidumpResult, attachment) => {
            minidumpFound = true;
            const minidumpProcess = minidumpResult.crashpadAnnotations?.process_type?.replace('-process', '');
            const event = await getEvent(minidumpProcess);
            if (minidumpResult.crashpadAnnotations) {
                // If we have crashpad annotations, we need to prepend their keys with 'crashpad.'
                // and add them to the Electron context
                const prependedAnnotations = Object.entries(minidumpResult.crashpadAnnotations).reduce((acc, [key, val]) => ((acc[`crashpad.${key}`] = val), acc), {});
                event.contexts = {
                    ...event.contexts,
                    electron: {
                        ...event.contexts?.electron,
                        ...prependedAnnotations,
                    },
                };
            }
            if (minidumpsRemaining > 0) {
                minidumpsRemaining -= 1;
                core.captureEvent(event, { attachments: [attachment] });
            }
        });
        return minidumpFound;
    }
    async function sendRendererCrash(client, options, contents, details) {
        const { getRendererName } = options;
        await sendNativeCrashes(client, (minidumpProcess) => {
            // We only call 'getRendererName' if this was in fact a renderer crash
            const crashedProcess = (minidumpProcess === 'renderer' && getRendererName ? getRendererName(contents) : minidumpProcess) || 'unknown';
            core.debug.log(`'${crashedProcess}' process '${details.reason}'`);
            return {
                contexts: {
                    electron: {
                        crashed_url: renderers.getRendererProperties(contents.id)?.url || 'unknown',
                        details,
                    },
                },
                level: 'fatal',
                // The default is javascript
                platform: 'native',
                tags: {
                    'event.environment': 'native',
                    'event.process': crashedProcess,
                    'exit.reason': details.reason,
                },
            };
        });
    }
    async function sendChildProcessCrash(client, details) {
        core.debug.log(`${details.type} process has ${details.reason}`);
        await sendNativeCrashes(client, (minidumpProcess) => ({
            contexts: {
                electron: { details },
            },
            level: 'fatal',
            // The default is javascript
            platform: 'native',
            tags: {
                'event.environment': 'native',
                'event.process': minidumpProcess || details.type,
                'exit.reason': details.reason,
            },
        }));
    }
    return {
        name: 'SentryMinidump',
        setup(client) {
            // Mac AppStore builds cannot run the crash reporter due to the sandboxing
            // requirements. In this case, we prevent enabling native crashes entirely.
            // https://electronjs.org/docs/tutorial/mac-app-store-submission-guide#limitations-of-mas-build
            if (process.mas) {
                return;
            }
            startCrashReporter();
            scopeStore = new store.BufferedWriteStore(electronNormalize.getSentryCachePath(), 'scope_v3', {
                scope: new core.Scope().getScopeData(),
            });
            scopeLastRun = scopeStore.get();
            try {
                minidumpLoader$1 = minidumpLoader.getMinidumpLoader();
            }
            catch (error) {
                // This is rare but we've seen reports:
                // https://github.com/getsentry/sentry-electron/issues/1102
                core.debug.error('Failed to create minidump loader', error);
            }
            const options = client.getOptions();
            setupScopeListener(client);
            if (!options?.dsn) {
                throw new Error('Attempted to enable Electron native crash reporter but no DSN was supplied');
            }
            renderers.trackRendererProperties();
            electron.app.on('render-process-gone', async (_, contents, details) => {
                if (electronNormalize.EXIT_REASONS.includes(details.reason)) {
                    await sendRendererCrash(client, options, contents, details);
                }
            });
            electron.app.on('child-process-gone', async (_, details) => {
                if (electronNormalize.EXIT_REASONS.includes(details.reason)) {
                    await sendChildProcessCrash(client, details);
                }
            });
            let sessionToRestore;
            // Start to submit recent minidump crashes. This will load breadcrumbs and
            // context information that was cached on disk in the previous app run, prior to the crash.
            sendNativeCrashes(client, async (minidumpProcess) => {
                const event = {
                    level: 'fatal',
                    platform: 'native',
                    tags: {
                        'event.environment': 'native',
                        'event.process': minidumpProcess || 'unknown',
                    },
                };
                // This crash was found at startup, we need to apply the scope and context from the previous run
                const previousRun = await scopeLastRun;
                if (previousRun) {
                    if (previousRun.scope) {
                        core.applyScopeDataToEvent(event, previousRun.scope);
                    }
                    event.release = previousRun.event?.release;
                    event.environment = previousRun.event?.environment;
                    event.contexts = previousRun.event?.contexts;
                    event.sdkProcessingMetadata = {
                        dynamicSamplingContext: {
                            trace_id: previousRun.scope.propagationContext.traceId,
                            release: previousRun.event?.release,
                            environment: previousRun.event?.environment,
                            public_key: client.getDsn()?.publicKey,
                        },
                    };
                }
                sessionToRestore = await sessions.setPreviousSessionAsCurrent();
                return event;
            })
                .then(async (minidumpsFound) => {
                if (!minidumpsFound) {
                    await sessions.previousSessionWasAbnormal();
                }
                else if (sessionToRestore) {
                    sessions.restorePreviousSession(sessionToRestore);
                }
            })
                .catch((error) => core.debug.error(error));
        },
    };
});

exports.sentryMinidumpIntegration = sentryMinidumpIntegration;//# sourceMappingURL=http://go/sourcemap/sourcemaps/bb70f115f7056b569146f670bfeb812a30eaf060/node_modules/@sentry/electron/main/integrations/sentry-minidump/index.js.map
