Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const core = require('@sentry/core');
const errorhandling = require('../utils/errorhandling.js');

const INTEGRATION_NAME = 'OnUnhandledRejection';

const _onUnhandledRejectionIntegration = ((options = {}) => {
  const opts = {
    mode: 'warn',
    ...options,
  } ;

  return {
    name: INTEGRATION_NAME,
    setup(client) {
      global.process.on('unhandledRejection', makeUnhandledPromiseHandler(client, opts));
    },
  };
}) ;

/**
 * Add a global promise rejection handler.
 */
const onUnhandledRejectionIntegration = core.defineIntegration(_onUnhandledRejectionIntegration);

/**
 * Send an exception with reason
 * @param reason string
 * @param promise promise
 *
 * Exported only for tests.
 */
function makeUnhandledPromiseHandler(
  client,
  options,
) {
  return function sendUnhandledPromise(reason, promise) {
    if (core.getClient() !== client) {
      return;
    }

    const level = options.mode === 'strict' ? 'fatal' : 'error';

    // this can be set in places where we cannot reliably get access to the active span/error
    // when the error bubbles up to this handler, we can use this to set the active span
    const activeSpanForError =
      reason && typeof reason === 'object' ? (reason )._sentry_active_span : undefined;

    const activeSpanWrapper = activeSpanForError
      ? (fn) => core.withActiveSpan(activeSpanForError, fn)
      : (fn) => fn();

    activeSpanWrapper(() => {
      core.captureException(reason, {
        originalException: promise,
        captureContext: {
          extra: { unhandledPromiseRejection: true },
          level,
        },
        mechanism: {
          handled: false,
          type: 'onunhandledrejection',
        },
      });
    });

    handleRejection(reason, options.mode);
  };
}

/**
 * Handler for `mode` option
 */
function handleRejection(reason, mode) {
  // https://github.com/nodejs/node/blob/7cf6f9e964aa00772965391c23acda6d71972a9a/lib/internal/process/promises.js#L234-L240
  const rejectionWarning =
    'This error originated either by ' +
    'throwing inside of an async function without a catch block, ' +
    'or by rejecting a promise which was not handled with .catch().' +
    ' The promise rejected with the reason:';

  /* eslint-disable no-console */
  if (mode === 'warn') {
    core.consoleSandbox(() => {
      console.warn(rejectionWarning);
      console.error(reason && typeof reason === 'object' && 'stack' in reason ? reason.stack : reason);
    });
  } else if (mode === 'strict') {
    core.consoleSandbox(() => {
      console.warn(rejectionWarning);
    });
    errorhandling.logAndExitProcess(reason);
  }
  /* eslint-enable no-console */
}

exports.makeUnhandledPromiseHandler = makeUnhandledPromiseHandler;
exports.onUnhandledRejectionIntegration = onUnhandledRejectionIntegration;//# sourceMappingURL=http://go/sourcemap/sourcemaps/b9e5948c1ad20443a5cecba6b84a3c9b99d62580/node_modules/@sentry/node-core/build/cjs/integrations/onunhandledrejection.js.map
