import { defineIntegration, captureException } from '@sentry/core';
import { dialog } from 'electron';

/** Capture unhandled errors. */
const onUncaughtExceptionIntegration = defineIntegration(() => {
    return {
        name: 'OnUncaughtException',
        setup(client) {
            const options = client.getOptions();
            global.process.on('uncaughtException', (error) => {
                captureException(error, {
                    originalException: error,
                    captureContext: {
                        level: 'fatal',
                    },
                    data: {
                        mechanism: {
                            handled: false,
                            type: 'generic',
                        },
                    },
                });
                client.flush(options.shutdownTimeout || 2000).then(() => {
                    if (options?.onFatalError) {
                        options.onFatalError(error);
                    }
                    else if (global.process.listenerCount('uncaughtException') <= 2) {
                        // In addition to this handler there is always one in Electron
                        // The dialog is only shown if there are no other handlers
                        // eslint-disable-next-line no-console
                        console.error('Uncaught Exception:');
                        // eslint-disable-next-line no-console
                        console.error(error);
                        const ref = error.stack;
                        const stack = ref !== undefined ? ref : `${error.name}: ${error.message}`;
                        const message = `Uncaught Exception:\n${stack}`;
                        dialog.showErrorBox('A JavaScript error occurred in the main process', message);
                    }
                }, () => {
                    // ignore
                });
            });
        },
    };
});

export { onUncaughtExceptionIntegration };//# sourceMappingURL=http://go/sourcemap/sourcemaps/b6e8aa9d472272da33220f2c22aaef5086384030/node_modules/@sentry/electron/esm/main/integrations/onuncaughtexception.js.map
