const node_child_process = require('node:child_process');
const core = require('@sentry/core');
const electron = require('electron');
const merge = require('../merge.js');

const DEFAULT_OPTIONS = {
    screen: true,
    deviceModelManufacturer: false,
};
function getWindowsDeviceModelManufacturer() {
    return new Promise((resolve) => {
        try {
            node_child_process.exec('powershell -NoProfile "Get-CimInstance -ClassName Win32_ComputerSystem | ConvertTo-Json"', (error, stdout) => {
                if (error) {
                    resolve({});
                }
                try {
                    const details = JSON.parse(stdout);
                    if (details.Manufacturer || details.Model) {
                        resolve({
                            manufacturer: details.Manufacturer,
                            model: details.Model,
                        });
                        return;
                    }
                }
                catch (_) {
                    //
                }
                resolve({});
            });
        }
        catch (_) {
            resolve({});
        }
    });
}
function getMacOSDeviceModelManufacturer() {
    return new Promise((resolve) => {
        try {
            node_child_process.exec('system_profiler SPHardwareDataType -json', (error, stdout) => {
                if (error) {
                    resolve({});
                }
                try {
                    const details = JSON.parse(stdout.trim());
                    if (details.SPHardwareDataType?.[0]?.machine_model) {
                        resolve({
                            manufacturer: 'Apple',
                            model: details.SPHardwareDataType[0].machine_model,
                        });
                        return;
                    }
                }
                catch (_) {
                    //
                }
                resolve({});
            });
        }
        catch (_) {
            resolve({});
        }
    });
}
function getDeviceModelManufacturer() {
    if (process.platform === 'win32') {
        return getWindowsDeviceModelManufacturer();
    }
    else if (process.platform === 'darwin') {
        return getMacOSDeviceModelManufacturer();
    }
    return Promise.resolve({});
}
/**
 * Adds additional Electron context to events
 */
const additionalContextIntegration = core.defineIntegration((userOptions = {}) => {
    const _lazyDeviceContext = {};
    let captureDeviceModelManufacturer = userOptions.deviceModelManufacturer;
    const options = {
        ...DEFAULT_OPTIONS,
        ...userOptions,
    };
    function setPrimaryDisplayInfo() {
        const display = electron.screen.getPrimaryDisplay();
        const width = Math.floor(display.size.width * display.scaleFactor);
        const height = Math.floor(display.size.height * display.scaleFactor);
        _lazyDeviceContext.screen_density = display.scaleFactor;
        _lazyDeviceContext.screen_resolution = `${width}x${height}`;
    }
    async function setDeviceModelManufacturer() {
        const { manufacturer, model } = await getDeviceModelManufacturer();
        if (manufacturer || model) {
            _lazyDeviceContext.manufacturer = manufacturer;
            _lazyDeviceContext.model = model;
        }
    }
    return {
        name: 'AdditionalContext',
        setup() {
            if (!options.screen) {
                return;
            }
            // Some metrics are only available after app ready so we lazily load them
            electron.app.whenReady().then(() => {
                setPrimaryDisplayInfo();
                electron.screen.on('display-metrics-changed', () => {
                    setPrimaryDisplayInfo();
                });
            }, () => {
                // ignore
            });
        },
        processEvent: async (event) => {
            if (captureDeviceModelManufacturer) {
                // Ensure we only fetch this once per session
                captureDeviceModelManufacturer = false;
                await setDeviceModelManufacturer();
            }
            return merge.mergeEvents(event, { contexts: { device: _lazyDeviceContext } });
        },
    };
});

exports.additionalContextIntegration = additionalContextIntegration;//# sourceMappingURL=http://go/sourcemap/sourcemaps/af6d64e4848e6185e482a2de5bac040191c8d790/node_modules/@sentry/electron/main/integrations/additional-context.js.map
