import { defineIntegration, LRUMap, forEachEnvelopeItem } from '@sentry/core';
import { app } from 'electron';
import { getDefaultReleaseName, getDefaultEnvironment } from '../context.js';
import { addHeaderToSession } from '../header-injection.js';
import { normaliseProfile } from '../normalize.js';

// A cache of renderer profiles which need attaching to events
let RENDERER_PROFILES;
/**
 * Caches a profile to later be re-attached to an event
 */
function rendererProfileFromIpc(event, profile) {
    if (!RENDERER_PROFILES) {
        return;
    }
    const profile_id = profile.event_id;
    RENDERER_PROFILES.set(profile_id, profile);
    if (event) {
        event.contexts = {
            ...event.contexts,
            // Re-add the profile context which we can later use to find the correct profile
            profile: {
                profile_id,
            },
        };
    }
}
/**
 * Injects 'js-profiling' document policy headers and ensures that profiles get forwarded with transactions
 */
const rendererProfilingIntegration = defineIntegration(() => {
    return {
        name: 'RendererProfiling',
        setup(client) {
            const options = client.getOptions();
            if (!options.enableRendererProfiling) {
                return;
            }
            RENDERER_PROFILES = new LRUMap(10);
            app.on('ready', () => {
                // Ensure the correct headers are set to enable the browser profiler
                options.getSessions().forEach((sesh) => addHeaderToSession(sesh, 'Document-Policy', 'js-profiling'));
            });
            // Copy the profiles back into the event envelopes
            client.on?.('beforeEnvelope', (envelope) => {
                let profile_id;
                forEachEnvelopeItem(envelope, (item, type) => {
                    if (type !== 'transaction') {
                        return;
                    }
                    for (let j = 1; j < item.length; j++) {
                        const event = item[j];
                        if (event?.contexts?.profile?.profile_id) {
                            profile_id = event.contexts.profile.profile_id;
                            // This can be removed as it's no longer needed
                            delete event.contexts.profile;
                        }
                    }
                });
                if (!profile_id) {
                    return;
                }
                const profile = RENDERER_PROFILES?.remove(profile_id);
                if (!profile) {
                    return;
                }
                normaliseProfile(profile, app.getAppPath());
                profile.release = options.release || getDefaultReleaseName();
                profile.environment = options.environment || getDefaultEnvironment();
                // @ts-expect-error untyped envelope
                envelope[1].push([{ type: 'profile' }, profile]);
            });
        },
    };
});

export { rendererProfileFromIpc, rendererProfilingIntegration };//# sourceMappingURL=http://go/sourcemap/sourcemaps/ac32b095dae9b8e0cfede6c5ebc55e589ee50e10/node_modules/@sentry/electron/esm/main/integrations/renderer-profiling.js.map
