import * as vscode from 'vscode';
import * as path from 'path';
import * as fs from 'fs';
import { MCPServer } from './mcpServer.js';
import { CursorIDEBrowserLogger } from './logger.js';

let mcpServer: MCPServer | undefined;

export function activate(context: vscode.ExtensionContext) {
	CursorIDEBrowserLogger.init();
	CursorIDEBrowserLogger.info('Cursor Browser Automation extension activated');

	// Register command to start MCP server
	context.subscriptions.push(
		vscode.commands.registerCommand('cursor.browserAutomation.start', async (preferredPort?: number, reuseExisting = true) => {
			try {
				if (!mcpServer) {
					CursorIDEBrowserLogger.info('Starting MCP server');
					mcpServer = new MCPServer(vscode.commands);
					const { port, reused, authToken } = await mcpServer.start(preferredPort, reuseExisting);
					CursorIDEBrowserLogger.info(`MCP server started on port ${port} (reused: ${reused})`);

					return { success: true, mcpPort: port, authToken, running: true, reused };
				}
				return {
					success: true,
					mcpPort: mcpServer.getPort(),
					authToken: mcpServer.getAuthToken(),
					running: true,
					reused: false
				};
			} catch (error) {
				CursorIDEBrowserLogger.error('Failed to start MCP server', error as Error);
				vscode.window.showErrorMessage(`Failed to start MCP server: ${error}`);
				return { success: false, error: String(error), running: false };
			}
		})
	);

	// Register command to stop MCP server
	context.subscriptions.push(
		vscode.commands.registerCommand('cursor.browserAutomation.stop', () => {
			if (mcpServer) {
				CursorIDEBrowserLogger.info('Stopping MCP server');
				mcpServer.stop();
				mcpServer = undefined;
				CursorIDEBrowserLogger.info('MCP server stopped');

				return { success: true, running: false };
			}
			return { success: true, running: false };
		})
	);

	// Register command to get MCP server status
	context.subscriptions.push(
		vscode.commands.registerCommand('cursor.browserAutomation.getStatus', () => {
			const status = {
				running: !!mcpServer,
				mcpPort: mcpServer?.getPort(),
				authToken: mcpServer?.getAuthToken()
			};
			return status;
		})
	);

	// Register command to pre-authorize a tabId (security: called by extension only, not HTTP)
	context.subscriptions.push(
		vscode.commands.registerCommand('cursor.browserAutomation.preAuthorizeTab', (tabId: string) => {
			if (!mcpServer) {
				return { success: false, error: 'MCP server not running' };
			}
			mcpServer.preAuthorizeTab(tabId);
			return { success: true, tabId };
		})
	);

}

export function deactivate() {
	mcpServer?.stop();
}