import { ExtensionContext } from 'vscode';
import { McpLogger } from './utils/logger';

/** Get a server-specific logger instance */
function getLogger(identifier: string) {
	return McpLogger.getLogger(identifier);
}
import * as vscode from 'vscode';
import { registerContext } from './commands/mcpCommands.js';
import { MCPOAuthClientProvider } from './commands/mcp/oauth.js';
import { auth } from '@modelcontextprotocol/sdk/client/auth.js';
import { EverythingProviderCreator } from './everythingProvider.js';


const deactivateTasks: { (): Promise<any> }[] = [];
// this method is called when vs code is activated
export function activate(context: ExtensionContext) {
	const everythingProviderCreator = new EverythingProviderCreator(context);
	deactivateTasks.push(async () => everythingProviderCreator.dispose());

	McpLogger.init();
	registerContext(context)

	// Register a URI handler to receive deep-link callbacks such as
	//  {protocol}://anysphere.cursor-mcp/oauth/<identifier>/callback?code=... (e.g., cursor://, cursor-nightly://, cursor-dev://)
	const uriHandlerDisp = vscode.window.registerUriHandler({
		handleUri: async (uri: vscode.Uri) => {
			try {
				// Expected path format: /oauth/<identifier>/callback
				const [, first, rawIdentifier, second] = uri.path.split('/');
				if (first !== 'oauth' || second !== 'callback' || !rawIdentifier) {
					vscode.window.showWarningMessage('Unrecognized deep link. Try updating Cursor');
					return;
				}
				// The identifier segment in the URL is percent-encoded, so we need to
				// decode it before using it for any look-ups (secrets storage, global
				// state, MCP service commands, …). All other parts of the extension
				// expect the *original* identifier that was generated by
				// `MCPService.computeIdentifier`.
				const identifier = decodeURIComponent(rawIdentifier);

				// Extract ?code=... but preserve URL-encoded format
				const params = new URLSearchParams(uri.query);
				const codeDecoded = params.get('code');
				if (!codeDecoded) {
					// Use general logger for cases without identifier
					McpLogger.warn('OAuth callback received without code parameter');
					return;
				}

				//  URLSearchParams.get() automatically decodes, but OAuth servers
				// (like Atlassian) expect the authorization code to remain URL-encoded.
				// so lets pull things out exactly from the url so we respect whatever the
				// server who sent it to us is expecting
				const codeMatch = uri.query.match(/(?:^|&)code=([^&]*)/);
				const code = codeMatch ? codeMatch[1] : codeDecoded;

				getLogger(identifier).info(`Received OAuth callback with code`);

				if (!code) {
					getLogger(identifier).error(`Authorization code not found in redirect url callback`);
					return;
				}

				// Retrieve stored server URL
				const serverUrl = context.globalState.get<string>(`[${identifier}] ${'mcp_server_url'}`);
				if (!serverUrl) {
					getLogger(identifier).error(`No stored server URL for OAuth flow`);
					return;
				}

				// Notify UI that we're now exchanging the token
				await vscode.commands.executeCommand('mcp.updateStatus', {
					identifier,
					status: { type: 'initializing' }
				});

				// Exchange the code for tokens via the SDK helper
				try {
					await auth(new MCPOAuthClientProvider(context, serverUrl, identifier, () => { }), {
						serverUrl,
						authorizationCode: code,
					});
					getLogger(identifier).info(`OAuth authorization completed`);
					// Clear status to allow reconnect
					await vscode.commands.executeCommand('mcp.reloadClient', { identifier, serverInfo: { type: 'streamableHttp', serverUrl } });
				} catch (e) {
					getLogger(identifier).error(`Failed to complete OAuth exchange`, e as Error);
				}
			} catch (err) {
				// Use general logger for general errors
				McpLogger.error('Error handling OAuth callback URI', err as Error);
			}
		}
	});
	deactivateTasks.push(async () => uriHandlerDisp.dispose());

}

export async function deactivate(): Promise<void> {
	for (const task of deactivateTasks) {
		await task();
	}
}
