const core = require('@sentry/core');
const node = require('@sentry/node');
const electron = require('electron');
const electronNormalize = require('../electron-normalize.js');

const DEFAULT_OPTIONS = {
    breadcrumbs: electronNormalize.EXIT_REASONS,
    events: ['abnormal-exit', 'launch-failed', 'integrity-failure'],
};
/** Gets message and severity */
function getMessageAndSeverity(reason, process) {
    const message = `'${process}' process exited with '${reason}'`;
    const messageFmt = node.logger.fmt `'${process}' process exited with '${reason}'`;
    switch (reason) {
        case 'abnormal-exit':
        case 'killed':
            return { message, level: 'warning', log: node.logger.warn, messageFmt };
        case 'crashed':
        case 'oom':
        case 'launch-failed':
        case 'integrity-failure':
            return { message, level: 'fatal', log: node.logger.error, messageFmt };
        default:
            return { message, level: 'debug', log: node.logger.info, messageFmt };
    }
}
/**
 * Adds breadcrumbs for:
 * - Electron child process events
 * - Node `child_process` events
 * - Node `worker_threads` events
 */
const childProcessIntegration = core.defineIntegration((userOptions = {}) => {
    const { breadcrumbs, events } = userOptions;
    const nodeIntegration = node.childProcessIntegration(userOptions);
    const options = {
        breadcrumbs: Array.isArray(breadcrumbs) ? breadcrumbs : breadcrumbs === false ? [] : DEFAULT_OPTIONS.breadcrumbs,
        events: Array.isArray(events) ? events : events === false ? [] : DEFAULT_OPTIONS.events,
    };
    return {
        name: 'ChildProcess',
        setup(client) {
            nodeIntegration.setup?.(client);
            const { breadcrumbs, events } = options;
            const allReasons = Array.from(new Set([...breadcrumbs, ...events]));
            // only hook these events if we're after more than just the unresponsive event
            if (allReasons.length > 0) {
                const clientOptions = client.getOptions();
                electron.app.on('child-process-gone', (_, details) => {
                    const { reason } = details;
                    const { message, level, log, messageFmt } = getMessageAndSeverity(details.reason, details.type);
                    // Capture message first
                    if (events.includes(reason)) {
                        core.captureMessage(message, { level, tags: { 'event.process': details.type } });
                    }
                    // And then add breadcrumbs for subsequent events
                    if (breadcrumbs.includes(reason)) {
                        core.addBreadcrumb({
                            type: 'process',
                            category: 'child-process',
                            message,
                            level,
                            data: details,
                        });
                        log(messageFmt, {
                            exitCode: details.exitCode,
                            name: details.name,
                            serviceName: details.serviceName,
                        });
                    }
                });
                electron.app.on('render-process-gone', (_, contents, details) => {
                    const { reason } = details;
                    const name = clientOptions?.getRendererName?.(contents) || 'renderer';
                    const { message, level, log, messageFmt } = getMessageAndSeverity(details.reason, name);
                    // Capture message first
                    if (events.includes(reason)) {
                        core.captureMessage(message, level);
                    }
                    // And then add breadcrumbs for subsequent events
                    if (breadcrumbs.includes(reason)) {
                        core.addBreadcrumb({
                            type: 'process',
                            category: 'child-process',
                            message,
                            level,
                            data: details,
                        });
                        log(messageFmt, {
                            exitCode: details.exitCode,
                        });
                    }
                });
            }
        },
    };
});

exports.childProcessIntegration = childProcessIntegration;//# sourceMappingURL=http://go/sourcemap/sourcemaps/9d178a4a5589981b62546448bb32920a8219a5d0/node_modules/@sentry/electron/main/integrations/child-process.js.map
