const core = require('@sentry/core');
const electron = require('electron');
const fs = require('fs');
const path = require('path');
const mutex = require('../../mutex.js');
const minidumpParser = require('./minidump-parser.js');

/** Maximum number of days to keep a minidump before deleting it. */
const MAX_AGE_DAYS = 30;
const MS_PER_DAY = 24 * 3600 * 1000;
/** Minimum number of milliseconds a minidump should not be modified for before we assume writing is complete */
const NOT_MODIFIED_MS = 1000;
const MAX_RETRY_MS = 5000;
const RETRY_DELAY_MS = 500;
const MAX_RETRIES = MAX_RETRY_MS / RETRY_DELAY_MS;
function delay(ms) {
    return new Promise((resolve) => setTimeout(resolve, ms));
}
/**
 * Creates a minidump loader
 * @param getMinidumpPaths A function that returns paths to minidumps
 * @param preProcessFile A function that pre-processes the minidump file
 * @returns A function to fetch minidumps
 */
function createMinidumpLoader(getMinidumpPaths) {
    // The mutex protects against a whole host of reentrancy issues and race conditions.
    const mutex$1 = new mutex.Mutex();
    return async (deleteAll, callback) => {
        // any calls to this function will be queued and run exclusively
        await mutex$1.runExclusive(async () => {
            for (const path$1 of await getMinidumpPaths()) {
                try {
                    if (deleteAll) {
                        continue;
                    }
                    core.debug.log('Found minidump', path$1);
                    let stats = await fs.promises.stat(path$1);
                    const thirtyDaysAgo = new Date().getTime() - MAX_AGE_DAYS * MS_PER_DAY;
                    if (stats.mtimeMs < thirtyDaysAgo) {
                        core.debug.log(`Ignoring minidump as it is over ${MAX_AGE_DAYS} days old`);
                        continue;
                    }
                    let retries = 0;
                    while (retries <= MAX_RETRIES) {
                        const twoSecondsAgo = new Date().getTime() - NOT_MODIFIED_MS;
                        if (stats.mtimeMs < twoSecondsAgo) {
                            const data = await fs.promises.readFile(path$1);
                            try {
                                const parsedMinidump = minidumpParser.parseMinidump(data);
                                core.debug.log('Sending minidump');
                                await callback(parsedMinidump, {
                                    attachmentType: 'event.minidump',
                                    filename: path.basename(path$1),
                                    data,
                                });
                            }
                            catch (e) {
                                const message = e instanceof Error ? e.toString() : 'Unknown error';
                                core.debug.warn(`Dropping minidump:\n${message}`);
                                break;
                            }
                            break;
                        }
                        core.debug.log(`Waiting. Minidump has been modified in the last ${NOT_MODIFIED_MS} milliseconds.`);
                        retries += 1;
                        await delay(RETRY_DELAY_MS);
                        // update the stats
                        stats = await fs.promises.stat(path$1);
                    }
                    if (retries >= MAX_RETRIES) {
                        core.debug.warn('Timed out waiting for minidump to stop being modified');
                    }
                }
                catch (e) {
                    core.debug.error('Failed to load minidump', e);
                }
                finally {
                    // We always attempt to delete the minidump
                    try {
                        await fs.promises.unlink(path$1);
                    }
                    catch (e) {
                        core.debug.warn('Could not delete minidump', path$1);
                    }
                }
            }
        });
    };
}
/** Attempts to remove the metadata file so Crashpad doesn't output `failed to stat report` errors to the console */
async function deleteCrashpadMetadataFile(crashesDirectory, waitMs = 100) {
    if (waitMs > 2000) {
        return;
    }
    const metadataPath = path.join(crashesDirectory, 'metadata');
    try {
        await fs.promises.unlink(metadataPath);
        core.debug.log('Deleted Crashpad metadata file', metadataPath);
    }
    catch (e) {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
        if (e.code && e.code == 'EBUSY') {
            // Since Crashpad probably still has the metadata file open, we make a few attempts to delete it, backing
            // off and waiting longer each time.
            setTimeout(async () => {
                await deleteCrashpadMetadataFile(crashesDirectory, waitMs * 2);
            }, waitMs);
        }
    }
}
async function readDirsAsync(paths) {
    const found = [];
    for (const path$1 of paths) {
        try {
            const files = await fs.promises.readdir(path$1);
            found.push(...files.map((file) => path.join(path$1, file)));
        }
        catch (_) {
            //
        }
    }
    return found;
}
/**
 * Gets the crashpad minidump loader
 */
function getMinidumpLoader() {
    const crashesDirectory = electron.app.getPath('crashDumps');
    const crashpadSubDirectory = process.platform === 'win32' ? 'reports' : 'completed';
    const dumpDirectories = [path.join(crashesDirectory, crashpadSubDirectory)];
    if (process.platform === 'darwin') {
        dumpDirectories.push(path.join(crashesDirectory, 'pending'));
    }
    return createMinidumpLoader(async () => {
        await deleteCrashpadMetadataFile(crashesDirectory).catch((error) => core.debug.error(error));
        const files = await readDirsAsync(dumpDirectories);
        return files.filter((file) => file.endsWith('.dmp'));
    });
}

exports.createMinidumpLoader = createMinidumpLoader;
exports.getMinidumpLoader = getMinidumpLoader;//# sourceMappingURL=http://go/sourcemap/sourcemaps/9b5f3f4f2368631e3455d37672ca61b6dce85430/node_modules/@sentry/electron/main/integrations/sentry-minidump/minidump-loader.js.map
