const core = require('@sentry/core');
const node = require('@sentry/node');
const electron = require('electron');

const getModuleFromFilename = node.createGetModuleFromFilename(electron.app.getAppPath());
/**
 * Normalizes all URLs in an event. See {@link normalizeUrl} for more
 * information. Mutates the passed in event.
 *
 * @param event The event to normalize.
 */
function normalizePaths(event, basePath) {
    // Retrieve stack traces and normalize their paths. Without this, grouping
    // would not work due to usernames in file paths.
    for (const exception of event.exception?.values || []) {
        for (const frame of exception.stacktrace?.frames || []) {
            if (frame.filename) {
                frame.filename = core.normalizeUrlToBase(frame.filename, basePath);
            }
        }
    }
    // We need to normalize debug ID images the same way as the stack frames for symbolicator to match them correctly
    for (const debugImage of event.debug_meta?.images || []) {
        if (debugImage.type === 'sourcemap') {
            debugImage.code_file = core.normalizeUrlToBase(debugImage.code_file, basePath);
        }
    }
    if (event.transaction) {
        event.transaction = core.normalizeUrlToBase(event.transaction, basePath);
    }
    const { request = {} } = event;
    if (request.url) {
        request.url = core.normalizeUrlToBase(request.url, basePath);
    }
    if (event.contexts?.feedback?.url && typeof event.contexts.feedback.url === 'string') {
        event.contexts.feedback.url = core.normalizeUrlToBase(event.contexts.feedback.url, basePath);
    }
    if (event.spans) {
        for (const span of event.spans) {
            if (span.description?.startsWith('file://')) {
                span.description = core.normalizeUrlToBase(span.description, basePath);
            }
        }
    }
    return event;
}
/** Normalizes URLs in any replay_event items found in an envelope */
function normalizeReplayEnvelope(options, envelope, basePath) {
    let modifiedEnvelope = core.createEnvelope(envelope[0]);
    let isReplay = false;
    core.forEachEnvelopeItem(envelope, (item, type) => {
        if (type === 'replay_event') {
            isReplay = true;
            const [headers, event] = item;
            const currentScope = core.getCurrentScope().getScopeData();
            event.breadcrumbs = currentScope.breadcrumbs;
            event.tags = currentScope.tags;
            event.user = currentScope.user;
            event.environment = options.environment;
            if (Array.isArray(event.urls)) {
                event.urls = event.urls.map((url) => core.normalizeUrlToBase(url, basePath));
            }
            if (event?.request?.url) {
                event.request.url = core.normalizeUrlToBase(event.request.url, basePath);
            }
            modifiedEnvelope = core.addItemToEnvelope(modifiedEnvelope, [headers, event]);
        }
        else if (type === 'replay_recording') {
            modifiedEnvelope = core.addItemToEnvelope(modifiedEnvelope, item);
        }
    });
    return isReplay ? modifiedEnvelope : envelope;
}
/**
 * Normalizes all URLs in a profile
 */
function normaliseProfile(profile, basePath) {
    for (const frame of profile.profile.frames) {
        if (frame.abs_path) {
            frame.abs_path = core.normalizeUrlToBase(frame.abs_path, basePath);
        }
        // filename isn't in the types but its in the actual data
        if ('filename' in frame && typeof frame.filename === 'string') {
            frame.filename = core.normalizeUrlToBase(frame.filename, basePath);
        }
        if (frame.module) {
            frame.module = getModuleFromFilename(frame.abs_path);
        }
    }
}

exports.normaliseProfile = normaliseProfile;
exports.normalizePaths = normalizePaths;
exports.normalizeReplayEnvelope = normalizeReplayEnvelope;//# sourceMappingURL=http://go/sourcemap/sourcemaps/9a5dd36e54f13fb9c0e74490ec44d080dbc5df50/node_modules/@sentry/electron/main/normalize.js.map
