Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const carrier = require('../carrier.js');
const debugBuild = require('../debug-build.js');
const worldwide = require('./worldwide.js');

/**
 * A Sentry Logger instance.
 *
 * @deprecated Use {@link debug} instead with the {@link SentryDebugLogger} type.
 */

const CONSOLE_LEVELS = [
  'debug',
  'info',
  'warn',
  'error',
  'log',
  'assert',
  'trace',
] ;

/** Prefix for logging strings */
const PREFIX = 'Sentry Logger ';

/** This may be mutated by the console instrumentation. */
const originalConsoleMethods

 = {};

/**
 * Temporarily disable sentry console instrumentations.
 *
 * @param callback The function to run against the original `console` messages
 * @returns The results of the callback
 */
function consoleSandbox(callback) {
  if (!('console' in worldwide.GLOBAL_OBJ)) {
    return callback();
  }

  const console = worldwide.GLOBAL_OBJ.console ;
  const wrappedFuncs = {};

  const wrappedLevels = Object.keys(originalConsoleMethods) ;

  // Restore all wrapped console methods
  wrappedLevels.forEach(level => {
    const originalConsoleMethod = originalConsoleMethods[level];
    wrappedFuncs[level] = console[level] ;
    console[level] = originalConsoleMethod ;
  });

  try {
    return callback();
  } finally {
    // Revert restoration to wrapped state
    wrappedLevels.forEach(level => {
      console[level] = wrappedFuncs[level] ;
    });
  }
}

function enable() {
  _getLoggerSettings().enabled = true;
}

function disable() {
  _getLoggerSettings().enabled = false;
}

function isEnabled() {
  return _getLoggerSettings().enabled;
}

function log(...args) {
  _maybeLog('log', ...args);
}

function info(...args) {
  _maybeLog('info', ...args);
}

function warn(...args) {
  _maybeLog('warn', ...args);
}

function error(...args) {
  _maybeLog('error', ...args);
}

function _debug(...args) {
  _maybeLog('debug', ...args);
}

function assert(...args) {
  _maybeLog('assert', ...args);
}

function trace(...args) {
  _maybeLog('trace', ...args);
}

function _maybeLog(level, ...args) {
  if (!debugBuild.DEBUG_BUILD) {
    return;
  }

  if (isEnabled()) {
    consoleSandbox(() => {
      worldwide.GLOBAL_OBJ.console[level](`${PREFIX}[${level}]:`, ...args);
    });
  }
}

function _getLoggerSettings() {
  if (!debugBuild.DEBUG_BUILD) {
    return { enabled: false };
  }

  return carrier.getGlobalSingleton('loggerSettings', () => ({ enabled: false }));
}

/**
 * This is a logger singleton which either logs things or no-ops if logging is not enabled.
 * The logger is a singleton on the carrier, to ensure that a consistent logger is used throughout the SDK.
 *
 * @deprecated Use {@link debug} instead.
 */
const logger = {
  /** Enable logging. */
  enable,
  /** Disable logging. */
  disable,
  /** Check if logging is enabled. */
  isEnabled,
  /** Log a message. */
  log,
  /** Log level info */
  info,
  /** Log a warning. */
  warn,
  /** Log an error. */
  error,
  /** Log a debug message. */
  debug: _debug,
  /** Log an assertion. */
  assert,
  /** Log a trace. */
  trace,
  // eslint-disable-next-line deprecation/deprecation
} ;

/**
 * This is a logger singleton which either logs things or no-ops if logging is not enabled.
 */
const debug = {
  /** Enable logging. */
  enable,
  /** Disable logging. */
  disable,
  /** Check if logging is enabled. */
  isEnabled,
  /** Log a message. */
  log,
  /** Log a warning. */
  warn,
  /** Log an error. */
  error,
} ;

exports.CONSOLE_LEVELS = CONSOLE_LEVELS;
exports.consoleSandbox = consoleSandbox;
exports.debug = debug;
exports.logger = logger;
exports.originalConsoleMethods = originalConsoleMethods;//# sourceMappingURL=http://go/sourcemap/sourcemaps/9675251a06b1314d50ff34b0cbe5109b78f848c0/node_modules/@sentry/core/build/cjs/utils/debug-logger.js.map
