import * as vscode from 'vscode';
import type {
	McpClient,
	McpLease,
	McpToolResult,
	NamedMcpToolDefinition,
} from '@anysphere/agent-exec';
import type { Context } from '@anysphere/context';
import { ExecutableMcpTool, ExecutableMcpToolSet, McpManager, type McpElicitationProviderFactory } from '@anysphere/local-exec';
import type { JsonValue } from '@bufbuild/protobuf';
import type { McpProvider } from '@cursor/types';
import { McpInstructions } from './proto/agent/v1/mcp_pb.js';
import { getMcpManager } from './commands/mcpCommands.js';
import { z } from 'zod';

// Schema for JSON Schema property definition
const JsonSchemaPropertySchema = z.object({
	description: z.string().optional(),
}).passthrough();

// Schema for JSON Schema parameters object
const JsonSchemaParametersSchema = z.object({
	properties: z.record(z.string(), JsonSchemaPropertySchema).optional(),
	required: z.array(z.string()).optional(),
}).passthrough();

export class VscodeMcpLease implements McpLease {
	private readonly mcpManager: McpManager;

	constructor(
		private readonly context: vscode.ExtensionContext,
	) {
		this.mcpManager = getMcpManager();

		this.initializeMcpProviders();

		this.context.subscriptions.push(
			vscode.cursor.onDidRegisterMcpProvider((provider) =>
				this.registerMcpProvider(provider)
			)
		);
	}

	async getClients(): Promise<Record<string, McpClient>> {
		return this.mcpManager.getClients();
	}

	async getClient(name: string): Promise<McpClient | undefined> {
		return this.mcpManager.getClient(name);
	}

	async getInstructions(ctx: Context): Promise<McpInstructions[]> {
		return this.mcpManager.getInstructions(ctx);
	}

	private async initializeMcpProviders(): Promise<void> {
		try {
			const providers = vscode.cursor.getAllMcpProviders();

			for (const provider of providers) {
				await this.registerMcpProviderToManager(this.mcpManager, provider);
			}
		} catch (error) {
			console.error('Failed to initialize MCP providers:', error);
		}
	}

	private async registerMcpProvider(provider: McpProvider): Promise<void> {
		try {
			await this.registerMcpProviderToManager(this.mcpManager, provider);
		} catch (error) {
			console.error(`Failed to register MCP provider ${provider.id}:`, error);
		}
	}

	private async registerMcpProviderToManager(
		mcpManager: McpManager,
		provider: McpProvider
	): Promise<void> {
		const clientWrapper: McpClient = {
			getState: async () => ({ kind: 'ready' as const }),
			getTools: async () => {
				const offerings = await provider.listOfferings();
				if (!offerings?.tools) {
					return [];
				}
				return offerings.tools.map((tool) => ({
					name: tool.name,
					description: tool.description,
					inputSchema: JSON.parse(tool.parameters),
				}));
			},
			callTool: async (
				toolName: string,
				args: Record<string, unknown>,
				_toolCallId?: string,
				_elicitationProvider?: any
			) => {
				return provider.callTool(toolName, args) as ReturnType<
					McpClient['callTool']
				>;
			},
			getInstructions: async (ctx: Context) => {
				return undefined;
			},
			listResources: async () => {
				const offerings = await provider.listOfferings();
				if (!offerings?.resources) {
					return { resources: [] };
				}
				return { resources: offerings.resources };
			},
			readResource: async (_args: { uri: string }) => {
				return { contents: [] };
			},
			listPrompts: async () => {
				const offerings = await provider.listOfferings();
				if (!offerings?.prompts) {
					return [];
				}
				return offerings.prompts.map((prompt) => {
					let argumentsList: Array<{ name: string; description?: string; required: boolean }> = [];

					if (prompt.parameters) {
						const parsed = JSON.parse(prompt.parameters);
						const result = JsonSchemaParametersSchema.safeParse(parsed);

						if (result.success && result.data.properties) {
							const requiredSet = new Set(result.data.required ?? []);
							argumentsList = Object.entries(result.data.properties).map(([name, prop]) => ({
								name,
								description: prop.description,
								required: requiredSet.has(name),
							}));
						}
					}

					return {
						name: prompt.name,
						description: prompt.description,
						arguments: argumentsList,
					};
				});
			},
			getPrompt: async (_name: string, _args?: Record<string, string>) => {
				return { messages: [] };
			},
			serverName: provider.id,
		};

		mcpManager.setClient(provider.id, clientWrapper);
	}

	async getToolSet(): Promise<ExecutableMcpToolSet> {
		const enabledServerIds = await vscode.cursor.getEnabledMcpServerIds();

		const allClients = await this.getClients();

		const enabledClients = Object.fromEntries(
			Object.entries(allClients).filter(([serverId]) =>
				enabledServerIds.includes(serverId)
			)
		);

		const clientTools = await Promise.all(
			Object.entries(enabledClients).map(([_, client]) =>
				client
					.getTools()
					.then((tools) =>
						tools.map((tool) => ({
							...tool,
							clientName: client.serverName,
							client: client,
						}))
					)
					.catch(() => [])
			)
		);

		const tools = clientTools.flat();

		const toolsMap: Record<string, ExecutableMcpTool> = {};
		for (const tool of tools) {
			toolsMap[`${tool.clientName}-${tool.name}`] = {
				definition: {
					...tool,
					providerIdentifier: tool.clientName,
					toolName: tool.name,
				},
				execute: async (
					args: Record<string, JsonValue>,
					toolCallId?: string,
					elicitationFactory?: McpElicitationProviderFactory
				) => {
					const elicitationProvider = elicitationFactory?.createProvider(
						tool.clientName,
						tool.name,
						toolCallId
					);

					const result = await tool.client.callTool(
						tool.name,
						args,
						toolCallId,
						elicitationProvider
					);
					return result;
				},
			};
		}

		return new ExecutableMcpToolSet(toolsMap);
	}

	async getTools(): Promise<NamedMcpToolDefinition[]> {
		const toolSet = await this.getToolSet();
		return toolSet.getTools();
	}
}
