/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Anysphere. All rights reserved.
 *--------------------------------------------------------------------------------------------*/

import * as vscode from 'vscode';
import { CursorIDEBrowserLogger } from './logger.js';

/**
 * Utility functions for building accessibility snapshots and interacting with browser pages
 */
const BROWSER_UTILS = `
// Helper to build accessibility snapshot
function buildPageSnapshot(depth = 0, maxDepth = 14) {
	function getTextFromIds(ids) {
		try {
			if (!ids) return '';
			const parts = [];
			ids.split(/\\s+/).forEach(id => {
				const el = document.getElementById(id);
				if (el) {
					const t = (el.innerText || el.textContent || '').trim();
					if (t) parts.push(t);
				}
			});
			return parts.join(' ').trim();
		} catch (_) { return ''; }
	}

	function getVisibleText(el) {
		try {
			const t = (el.innerText || '').trim();
			if (t) return t;
			const textNodes = Array.from(el.childNodes).filter(n => n.nodeType === Node.TEXT_NODE);
			const direct = textNodes.map(n => (n.textContent || '').trim()).filter(Boolean).join(' ');
			return direct.trim();
		} catch (_) { return ''; }
	}

	function computeAccessibleName(el, roleAttr) {
		try {
			const labelledBy = el.getAttribute('aria-labelledby');
			const fromLabelledBy = getTextFromIds(labelledBy);
			if (fromLabelledBy) return fromLabelledBy.substring(0, 200);

			const tag = el.tagName.toLowerCase();
			const interactiveRoles = new Set(['button', 'link', 'menuitem', 'option', 'tab', 'checkbox', 'radio']);
			const prefersVisible = interactiveRoles.has((roleAttr || '').toLowerCase()) || ['a','button','label','summary','option','time','li','span','div','p','h1','h2','h3','h4','h5','h6'].includes(tag);
			if (prefersVisible) {
				const visible = getVisibleText(el);
				if (visible) return visible.substring(0, 200);
			}

			const ariaLabel = (el.getAttribute('aria-label') || '').trim();
			if (ariaLabel) return ariaLabel.substring(0, 200);

			if (tag === 'img') {
				const alt = (el.getAttribute('alt') || '').trim();
				if (alt) return alt.substring(0, 200);
			}
			if (tag === 'input') {
				const type = (el.getAttribute('type') || '').toLowerCase();
				const value = el.value || el.getAttribute('value') || '';
				const placeholder = (el.getAttribute('placeholder') || '').trim();
				if (type === 'button' || type === 'submit' || type === 'reset') {
					if (value) return String(value).substring(0, 200);
				}
				if (placeholder) return placeholder.substring(0, 200);
				if (value) return String(value).substring(0, 200);
			}
			if (tag === 'textarea' || tag === 'select' || tag === 'option') {
				const val = el.value || '';
				const vis = getVisibleText(el);
				return (vis || val).substring(0, 200);
			}

			const title = (el.getAttribute('title') || '').trim();
			if (title) return title.substring(0, 200);
			const placeholder = (el.getAttribute('placeholder') || '').trim();
			if (placeholder) return placeholder.substring(0, 200);

			const last = getVisibleText(el);
			return last.substring(0, 200);
		} catch (_) { return ''; }
	}

	function buildAccessibilityTree(element, depth = 0, maxDepth = 14) {
		if (depth > maxDepth) return null;

		let ref = element.getAttribute('data-cursor-ref');
		if (!ref) {
			ref = 'ref-' + Math.random().toString(36).substring(2, 15);
			element.setAttribute('data-cursor-ref', ref);
		}

		const roleAttr = element.getAttribute('role') || '';
		const name = computeAccessibleName(element, roleAttr);

		const node = {
			ref,
			role: roleAttr || element.tagName.toLowerCase(),
			name: name,
			tag: element.tagName.toLowerCase(),
			children: []
		};

		const meaningfulTags = ['a', 'button', 'input', 'select', 'textarea', 'img', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'nav', 'main', 'header', 'footer', 'section', 'article', 'form', 'label', 'div', 'span', 'ul', 'ol', 'li', 'p', 'strong', 'em', 'small', 'time', 'option'];

		for (const child of element.children) {
			const childTag = child.tagName.toLowerCase();
			const shouldInclude = meaningfulTags.includes(childTag) ||
				child.getAttribute('role') ||
				child.getAttribute('aria-label') ||
				child.querySelector('a, button, input, select, textarea, time, option, [role="button"], [role="link"], [role="menuitem"], [role="tab"], [role="gridcell"], [role="cell"], [role="row"]');

			if (shouldInclude) {
				const childNode = buildAccessibilityTree(child, depth + 1, maxDepth);
				if (childNode) {
					node.children.push(childNode);
				}
			}
		}

		return node;
	}

	return buildAccessibilityTree(document.body, depth, maxDepth);
}
`;

/**
 * Execute JavaScript in the browser view
 */
async function executeInBrowser(code: string): Promise<unknown> {
	try {
		return await vscode.commands.executeCommand('cursor.browserView.executeJavaScript', code);
	} catch (error) {
		CursorIDEBrowserLogger.error('Failed to execute JavaScript in browser:', error);
		throw error;
	}
}

/**
 * Browser tool implementations using direct JavaScript execution
 */
export const BrowserTools = {
	async navigate(params: { url: string }): Promise<unknown> {
		CursorIDEBrowserLogger.info(`Navigating to ${params.url}`);
		await vscode.commands.executeCommand('cursor.browserView.navigate', params.url);

		// Wait for page to load and return snapshot
		await new Promise(resolve => setTimeout(resolve, 1000));
		return await BrowserTools.snapshot({});
	},

	async snapshot(_params: Record<string, unknown>): Promise<unknown> {
		const code = `
			${BROWSER_UTILS}
			(function() {
				const tree = buildPageSnapshot();
				return {
					action: 'snapshot',
					success: true,
					pageState: {
						url: window.location.href,
						title: document.title,
						snapshot: tree
					}
				};
			})();
		`;
		return await executeInBrowser(code);
	},

	async click(params: { ref: string; element?: string; doubleClick?: boolean; button?: string; modifiers?: string[] }): Promise<unknown> {
		const code = `
			${BROWSER_UTILS}
			(function() {
				const ref = ${JSON.stringify(params.ref)};
				const element = document.querySelector('[data-cursor-ref="' + ref + '"]');
				if (!element) throw new Error('Element not found');

				const rect = element.getBoundingClientRect();
				const cx = Math.round(rect.left + rect.width / 2);
				const cy = Math.round(rect.top + rect.height / 2);

				// Scroll into view if needed
				if (rect.top < 0 || rect.left < 0 || rect.bottom > window.innerHeight || rect.right > window.innerWidth) {
					element.scrollIntoView({ block: 'center', inline: 'center', behavior: 'auto' });
				}

				const buttonValue = ${JSON.stringify(params.button)} === 'right' ? 2 :
					${JSON.stringify(params.button)} === 'middle' ? 1 : 0;

				const modifiers = ${JSON.stringify(params.modifiers || [])};
				const mouseEventOptions = {
					bubbles: true,
					cancelable: true,
					view: window,
					button: buttonValue,
					buttons: 1 << buttonValue,
					ctrlKey: modifiers.includes('Control') || modifiers.includes('ControlOrMeta'),
					shiftKey: modifiers.includes('Shift'),
					altKey: modifiers.includes('Alt'),
					metaKey: modifiers.includes('Meta') || modifiers.includes('ControlOrMeta'),
					clientX: cx,
					clientY: cy
				};

				if (element.focus) element.focus();

				element.dispatchEvent(new MouseEvent('mousedown', mouseEventOptions));
				element.dispatchEvent(new MouseEvent('mouseup', mouseEventOptions));
				element.dispatchEvent(new MouseEvent('click', mouseEventOptions));

				if (${JSON.stringify(params.doubleClick)}) {
					element.dispatchEvent(new MouseEvent('mousedown', mouseEventOptions));
					element.dispatchEvent(new MouseEvent('mouseup', mouseEventOptions));
					element.dispatchEvent(new MouseEvent('click', mouseEventOptions));
					element.dispatchEvent(new MouseEvent('dblclick', mouseEventOptions));
				}

				const snapshot = buildPageSnapshot();
				return {
					action: 'click',
					success: true,
					details: {
						doubleClick: ${JSON.stringify(params.doubleClick)} || false,
						button: ${JSON.stringify(params.button)} || 'left'
					},
					pageState: {
						url: window.location.href,
						title: document.title,
						snapshot: snapshot
					}
				};
			})();
		`;
		return await executeInBrowser(code);
	},

	async type(params: { ref: string; text: string; element?: string; submit?: boolean; slowly?: boolean }): Promise<unknown> {
		const code = `
			${BROWSER_UTILS}
			(function() {
				const ref = ${JSON.stringify(params.ref)};
				const element = document.querySelector('[data-cursor-ref="' + ref + '"]');
				if (!element) throw new Error('Element not found');

				element.focus();
				const text = ${JSON.stringify(params.text)};
				const slowly = ${JSON.stringify(params.slowly)} || false;
				const submit = ${JSON.stringify(params.submit)} || false;

				const isContentEditable = element.isContentEditable;

				if (slowly) {
					// Type one character at a time
					const delay = 50;
					for (let i = 0; i < text.length; i++) {
						const char = text[i];
						if (isContentEditable) {
							const selection = window.getSelection();
							const range = document.createRange();
							range.selectNodeContents(element);
							range.collapse(false);
							selection.removeAllRanges();
							selection.addRange(range);
							document.execCommand('insertText', false, char);
						} else {
							element.value = text.substring(0, i + 1);
						}
						element.dispatchEvent(new KeyboardEvent('keydown', { key: char, bubbles: true, cancelable: true }));
						element.dispatchEvent(new Event('input', { bubbles: true }));
						element.dispatchEvent(new KeyboardEvent('keyup', { key: char, bubbles: true, cancelable: true }));
						// Small delay between characters
						const startTime = Date.now();
						while (Date.now() - startTime < delay) { /* busy wait */ }
					}
				} else {
					if (isContentEditable) {
						element.textContent = text;
					} else {
						element.value = text;
					}
					element.dispatchEvent(new Event('input', { bubbles: true }));
				}

				element.dispatchEvent(new Event('change', { bubbles: true }));

				if (submit) {
					element.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', keyCode: 13, code: 'Enter', which: 13, bubbles: true, cancelable: true }));
					element.dispatchEvent(new KeyboardEvent('keypress', { key: 'Enter', keyCode: 13, code: 'Enter', which: 13, bubbles: true, cancelable: true }));
					element.dispatchEvent(new KeyboardEvent('keyup', { key: 'Enter', keyCode: 13, code: 'Enter', which: 13, bubbles: true, cancelable: true }));
				}

				const snapshot = buildPageSnapshot();
				return {
					action: 'type',
					success: true,
					details: { slowly, submitted: submit },
					pageState: {
						url: window.location.href,
						title: document.title,
						snapshot: snapshot
					}
				};
			})();
		`;
		return await executeInBrowser(code);
	},

	async hover(params: { ref: string; element?: string }): Promise<unknown> {
		const code = `
			${BROWSER_UTILS}
			(function() {
				const ref = ${JSON.stringify(params.ref)};
				const element = document.querySelector('[data-cursor-ref="' + ref + '"]');
				if (!element) throw new Error('Element not found');

				const rect = element.getBoundingClientRect();
				const hx = Math.round(rect.left + rect.width / 2);
				const hy = Math.round(rect.top + rect.height / 2);

				const mouseEventOptions = {
					bubbles: true,
					cancelable: true,
					view: window
				};

				element.dispatchEvent(new MouseEvent('mouseenter', { ...mouseEventOptions, clientX: hx, clientY: hy }));
				element.dispatchEvent(new MouseEvent('mouseover', { ...mouseEventOptions, clientX: hx, clientY: hy }));
				element.dispatchEvent(new MouseEvent('mousemove', { ...mouseEventOptions, clientX: hx, clientY: hy }));

				const snapshot = buildPageSnapshot();
				return {
					action: 'hover',
					success: true,
					pageState: {
						url: window.location.href,
						title: document.title,
						snapshot: snapshot
					}
				};
			})();
		`;
		return await executeInBrowser(code);
	},

	async selectOption(params: { ref: string; values: string[]; element?: string }): Promise<unknown> {
		const code = `
			${BROWSER_UTILS}
			(function() {
				const ref = ${JSON.stringify(params.ref)};
				const element = document.querySelector('[data-cursor-ref="' + ref + '"]');
				if (!element) throw new Error('Element not found');

				const selectElement = element;
				const values = ${JSON.stringify(params.values)};

				if (!selectElement.multiple) {
					selectElement.value = '';
				} else {
					Array.from(selectElement.options).forEach(option => {
						option.selected = false;
					});
				}

				const selectedValues = [];
				for (const value of values) {
					let optionFound = false;
					for (const option of selectElement.options) {
						if (option.value === value) {
							option.selected = true;
							selectedValues.push(value);
							optionFound = true;
							break;
						}
					}
					if (!optionFound) {
						throw new Error('Option with value "' + value + '" not found');
					}
				}

				selectElement.dispatchEvent(new Event('input', { bubbles: true }));
				selectElement.dispatchEvent(new Event('change', { bubbles: true }));

				const snapshot = buildPageSnapshot();
				return {
					action: 'select_option',
					success: true,
					details: { selectedValues },
					pageState: {
						url: window.location.href,
						title: document.title,
						snapshot: snapshot
					}
				};
			})();
		`;
		return await executeInBrowser(code);
	},

	async pressKey(params: { key: string }): Promise<unknown> {
		const code = `
			(function() {
				const key = ${JSON.stringify(params.key)};
				const activeElement = document.activeElement || document.body;

				activeElement.dispatchEvent(new KeyboardEvent('keydown', { key, bubbles: true, cancelable: true }));
				activeElement.dispatchEvent(new KeyboardEvent('keypress', { key, bubbles: true, cancelable: true }));
				activeElement.dispatchEvent(new KeyboardEvent('keyup', { key, bubbles: true, cancelable: true }));

				return { action: 'press_key', success: true, key };
			})();
		`;
		return await executeInBrowser(code);
	},

	async waitFor(params: { time?: number; text?: string; textGone?: string }): Promise<unknown> {
		if (params.time !== undefined) {
			await new Promise(resolve => setTimeout(resolve, params.time! * 1000));
			return { action: 'wait_for', success: true, type: 'time', duration: params.time };
		}

		const code = `
			(function() {
				const targetText = ${JSON.stringify(params.text || params.textGone)};
				const waitForDisappear = ${JSON.stringify(!!params.textGone)};
				const timeout = 30000;
				const startTime = Date.now();

				return new Promise((resolve) => {
					const checkInterval = setInterval(() => {
						const bodyText = document.body.innerText || document.body.textContent || '';
						const found = bodyText.includes(targetText);

						if (waitForDisappear ? !found : found) {
							clearInterval(checkInterval);
							resolve({
								action: 'wait_for',
								success: true,
								type: waitForDisappear ? 'text_gone' : 'text_appear',
								text: targetText
							});
						}

						if (Date.now() - startTime > timeout) {
							clearInterval(checkInterval);
							resolve({
								action: 'wait_for',
								success: false,
								error: 'Timeout waiting for text',
								type: waitForDisappear ? 'text_gone' : 'text_appear',
								text: targetText
							});
						}
					}, 500);
				});
			})();
		`;
		return await executeInBrowser(code);
	},

	async consoleMessages(_params: Record<string, unknown>): Promise<unknown> {
		// Console logs are now tracked by the main process
		const logs = await vscode.commands.executeCommand<Array<{ type: string; message: string; timestamp: number }>>('cursor.browserView.getConsoleLogs');
		return {
			action: 'console_messages',
			success: true,
			messages: logs
		};
	},

	async networkRequests(_params: Record<string, unknown>): Promise<unknown> {
		// Network requests are now tracked by the main process
		const requests = await vscode.commands.executeCommand<Array<{
			url: string;
			method: string;
			statusCode?: number;
			timestamp: number;
			resourceType?: string;
		}>>('cursor.browserView.getNetworkRequests');
		return {
			action: 'network_requests',
			success: true,
			requests
		};
	},

	async takeScreenshot(params: { filename?: string; type?: string; fullPage?: boolean; element?: string; ref?: string }): Promise<unknown> {
		// Screenshot will be handled by the main process
		return await vscode.commands.executeCommand('cursor.browserView.takeScreenshot', params);
	},

	async goBack(_params: Record<string, unknown>): Promise<unknown> {
		return await vscode.commands.executeCommand('cursor.browserView.goBack');
	},

	async resize(params: { width: number; height: number }): Promise<unknown> {
		return await vscode.commands.executeCommand('cursor.browserView.resize', params);
	}
};

