import { defineIntegration, getClient, fill, LRUMap, startInactiveSpan, SentryNonRecordingSpan, SEMANTIC_ATTRIBUTE_SENTRY_ORIGIN, getTraceData, debug, setHttpStatus, getBreadcrumbLogLevelFromHttpStatusCode, addBreadcrumb, stringMatchesSomePattern } from '@sentry/core';
import { logger } from '@sentry/node';
import { net } from 'electron';
import * as urlModule from 'url';

/**
 * Trimmed down version of the code from Electron here:
 * https://github.com/electron/electron/blob/f3df76dbdc58cb704637b89357e1400791c92cfe/lib/browser/api/net.ts#L209-L269
 *
 * We want to match the final URL that Electron uses
 */
function parseOptions(optionsIn) {
    const { method, options } = typeof optionsIn === 'string'
        ? // eslint-disable-next-line deprecation/deprecation
            { method: 'GET', options: urlModule.parse(optionsIn) }
        : { method: (optionsIn.method || 'GET').toUpperCase(), options: optionsIn };
    let url = 'url' in options ? options.url : undefined;
    if (!url) {
        const urlObj = {};
        urlObj.protocol = options.protocol || 'http:';
        if (options.host) {
            urlObj.host = options.host;
        }
        else {
            if (options.hostname) {
                urlObj.hostname = options.hostname;
            }
            else {
                urlObj.hostname = 'localhost';
            }
            if (options.port) {
                urlObj.port = options.port;
            }
        }
        // eslint-disable-next-line deprecation/deprecation
        const pathObj = urlModule.parse(options.path || '/');
        urlObj.pathname = pathObj.pathname;
        urlObj.search = pathObj.search;
        urlObj.hash = pathObj.hash;
        url = urlModule.format(urlObj);
    }
    return {
        method,
        url,
    };
}
function createWrappedRequestFactory(options, tracePropagationTargets) {
    // We're caching results so we don't have to recompute regexp every time we create a request.
    const createSpanUrlMap = new LRUMap(100);
    const headersUrlMap = new LRUMap(100);
    const shouldCreateSpan = (method, url) => {
        if (options.tracing === undefined) {
            return true;
        }
        if (options.tracing === false) {
            return false;
        }
        const key = `${method}:${url}`;
        const cachedDecision = createSpanUrlMap.get(key);
        if (cachedDecision !== undefined) {
            return cachedDecision;
        }
        const decision = options.tracing === true || options.tracing(method, url);
        createSpanUrlMap.set(key, decision);
        return decision;
    };
    // This will be considerably simpler once `tracingOrigins` is removed in the next major release
    const shouldAttachTraceData = (method, url) => {
        const key = `${method}:${url}`;
        const cachedDecision = headersUrlMap.get(key);
        if (cachedDecision !== undefined) {
            return cachedDecision;
        }
        if (tracePropagationTargets) {
            const decision = stringMatchesSomePattern(url, tracePropagationTargets);
            headersUrlMap.set(key, decision);
            return decision;
        }
        // We cannot reach here since either `tracePropagationTargets` or `tracingOrigins` will be defined but TypeScript
        // cannot infer that
        return true;
    };
    return function wrappedRequestMethodFactory(originalRequestMethod) {
        return function requestMethod(reqOptions) {
            const { url, method } = parseOptions(reqOptions);
            const request = originalRequestMethod.apply(this, [reqOptions]);
            if (url.match(/sentry_key/) || request.getHeader('x-sentry-auth')) {
                return request;
            }
            const span = shouldCreateSpan(method, url)
                ? startInactiveSpan({
                    name: `${method} ${url}`,
                    onlyIfParent: true,
                    attributes: {
                        url,
                        type: 'net.request',
                        'http.method': method,
                    },
                    op: 'http.client',
                })
                : new SentryNonRecordingSpan();
            span.setAttribute(SEMANTIC_ATTRIBUTE_SENTRY_ORIGIN, 'auto.http.electron.net');
            if (shouldAttachTraceData(method, url)) {
                for (const [key, value] of Object.entries(getTraceData({ span }))) {
                    debug.log(`[Tracing] Adding ${key} header ${value} to outgoing request to "${url}": `);
                    request.setHeader(key, value);
                }
            }
            return request
                .once('response', function (res) {
                if (options.breadcrumbs !== false) {
                    addRequestBreadcrumb('response', method, url, this, res);
                }
                if (res.statusCode) {
                    setHttpStatus(span, res.statusCode);
                }
                span.end();
            })
                .once('error', function (_error) {
                if (options.breadcrumbs !== false) {
                    addRequestBreadcrumb('error', method, url, this, undefined);
                }
                setHttpStatus(span, 500);
                span.end();
            });
        };
    };
}
/**
 * Captures Breadcrumb based on provided request/response pair
 */
function addRequestBreadcrumb(event, method, url, req, res) {
    const level = getBreadcrumbLogLevelFromHttpStatusCode(res?.statusCode);
    addBreadcrumb({
        type: 'http',
        category: 'electron.net',
        data: {
            url,
            method: method,
            status_code: res?.statusCode,
        },
        level,
    }, {
        event,
        request: req,
        response: res,
    });
    const attributes = { statusCode: res?.statusCode };
    switch (level) {
        case 'error':
            logger.error(logger.fmt `Electron.net request failed: ${method} ${url}`, attributes);
            break;
        case 'warning':
            logger.warn(logger.fmt `Electron.net request warning: ${method} ${url}`, attributes);
            break;
        default:
            logger.info(logger.fmt `Electron.net request succeeded: ${method} ${url}`, attributes);
    }
}
/**
 * Electron 'net' module integration
 */
const electronNetIntegration = defineIntegration((options = {}) => {
    return {
        name: 'ElectronNet',
        setup() {
            const clientOptions = getClient()?.getOptions();
            // No need to instrument if we don't want to track anything
            if (options.breadcrumbs === false && options.tracing === false) {
                return;
            }
            fill(net, 'request', createWrappedRequestFactory(options, clientOptions?.tracePropagationTargets));
        },
    };
});

export { electronNetIntegration };//# sourceMappingURL=http://go/sourcemap/sourcemaps/8cd6f62c4c264240eb1925773707665d0f5b25e0/node_modules/@sentry/electron/esm/main/integrations/net-breadcrumbs.js.map
