import { exec } from 'node:child_process';
import { defineIntegration } from '@sentry/core';
import { app, screen } from 'electron';
import { mergeEvents } from '../merge.js';

const DEFAULT_OPTIONS = {
    screen: true,
    deviceModelManufacturer: false,
};
function getWindowsDeviceModelManufacturer() {
    return new Promise((resolve) => {
        try {
            exec('powershell -NoProfile "Get-CimInstance -ClassName Win32_ComputerSystem | ConvertTo-Json"', (error, stdout) => {
                if (error) {
                    resolve({});
                }
                try {
                    const details = JSON.parse(stdout);
                    if (details.Manufacturer || details.Model) {
                        resolve({
                            manufacturer: details.Manufacturer,
                            model: details.Model,
                        });
                        return;
                    }
                }
                catch (_) {
                    //
                }
                resolve({});
            });
        }
        catch (_) {
            resolve({});
        }
    });
}
function getMacOSDeviceModelManufacturer() {
    return new Promise((resolve) => {
        try {
            exec('system_profiler SPHardwareDataType -json', (error, stdout) => {
                if (error) {
                    resolve({});
                }
                try {
                    const details = JSON.parse(stdout.trim());
                    if (details.SPHardwareDataType?.[0]?.machine_model) {
                        resolve({
                            manufacturer: 'Apple',
                            model: details.SPHardwareDataType[0].machine_model,
                        });
                        return;
                    }
                }
                catch (_) {
                    //
                }
                resolve({});
            });
        }
        catch (_) {
            resolve({});
        }
    });
}
function getDeviceModelManufacturer() {
    if (process.platform === 'win32') {
        return getWindowsDeviceModelManufacturer();
    }
    else if (process.platform === 'darwin') {
        return getMacOSDeviceModelManufacturer();
    }
    return Promise.resolve({});
}
/**
 * Adds additional Electron context to events
 */
const additionalContextIntegration = defineIntegration((userOptions = {}) => {
    const _lazyDeviceContext = {};
    let captureDeviceModelManufacturer = userOptions.deviceModelManufacturer;
    const options = {
        ...DEFAULT_OPTIONS,
        ...userOptions,
    };
    function setPrimaryDisplayInfo() {
        const display = screen.getPrimaryDisplay();
        const width = Math.floor(display.size.width * display.scaleFactor);
        const height = Math.floor(display.size.height * display.scaleFactor);
        _lazyDeviceContext.screen_density = display.scaleFactor;
        _lazyDeviceContext.screen_resolution = `${width}x${height}`;
    }
    async function setDeviceModelManufacturer() {
        const { manufacturer, model } = await getDeviceModelManufacturer();
        if (manufacturer || model) {
            _lazyDeviceContext.manufacturer = manufacturer;
            _lazyDeviceContext.model = model;
        }
    }
    return {
        name: 'AdditionalContext',
        setup() {
            if (!options.screen) {
                return;
            }
            // Some metrics are only available after app ready so we lazily load them
            app.whenReady().then(() => {
                setPrimaryDisplayInfo();
                screen.on('display-metrics-changed', () => {
                    setPrimaryDisplayInfo();
                });
            }, () => {
                // ignore
            });
        },
        processEvent: async (event) => {
            if (captureDeviceModelManufacturer) {
                // Ensure we only fetch this once per session
                captureDeviceModelManufacturer = false;
                await setDeviceModelManufacturer();
            }
            return mergeEvents(event, { contexts: { device: _lazyDeviceContext } });
        },
    };
});

export { additionalContextIntegration };//# sourceMappingURL=http://go/sourcemap/sourcemaps/6b030be1a256f7464cbd98e48627a802a4831bf0/node_modules/@sentry/electron/esm/main/integrations/additional-context.js.map
