const core = require('@sentry/core');
const node = require('@sentry/node');
const electron = require('electron');
const electronNormalize = require('./electron-normalize.js');
const store = require('./store.js');

const PERSIST_INTERVAL_MS = 60000;
/** Stores the app session in case of termination due to main process crash or app killed */
let sessionStore;
/** Previous session if it did not exit cleanly */
let previousSession;
function getSessionStore() {
    if (!sessionStore) {
        sessionStore = new store.Store(electronNormalize.getSentryCachePath(), 'session', undefined);
        previousSession = sessionStore.get().then((sesh) => (sesh ? core.makeSession(sesh) : sesh));
    }
    return sessionStore;
}
/** Copies a session and removes the toJSON function so it can be serialised without conversion */
function makeSessionSafeToSerialize(session) {
    const copy = { ...session };
    // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
    delete copy.toJSON;
    return copy;
}
let persistTimer;
/** Starts a session */
function startSession(sendOnCreate) {
    const session = core.startSession();
    if (sendOnCreate) {
        core.captureSession();
    }
    getSessionStore()
        .set(makeSessionSafeToSerialize(session))
        .catch(() => {
        // Does not throw
    });
    // Every PERSIST_INTERVAL, write the session to disk
    persistTimer = setInterval(async () => {
        const currentSession = core.getIsolationScope().getSession();
        // Only bother saving if it hasn't already ended
        if (currentSession && currentSession.status === 'ok') {
            await getSessionStore().set(makeSessionSafeToSerialize(currentSession));
        }
    }, PERSIST_INTERVAL_MS);
}
/** Cleanly ends a session */
async function endSession() {
    // Once the session had ended there is no point persisting it
    if (persistTimer) {
        clearInterval(persistTimer);
    }
    const session = core.getIsolationScope().getSession();
    if (session) {
        if (session.status === 'ok') {
            core.debug.log('Ending session');
            core.endSession();
        }
        else {
            core.debug.log('Session was already ended');
        }
    }
    else {
        core.debug.log('No session');
    }
    await getSessionStore().clear();
    await node.flush(2000);
}
/** Determines if a Date is likely to have occurred in the previous uncompleted session */
async function unreportedDuringLastSession(crashDate) {
    if (!crashDate) {
        return false;
    }
    const previousSessionModified = await getSessionStore().getModifiedDate();
    // There is no previous session
    if (previousSessionModified === undefined) {
        return false;
    }
    const previousSessionModifiedTime = previousSessionModified.getTime();
    const crashTime = crashDate.getTime();
    // Session could have run until modified time + persist interval
    const prevSessionEnd = previousSessionModifiedTime + PERSIST_INTERVAL_MS;
    // Event cannot have occurred before last persist time, We add a 2 second overlap to be sure
    const lastPersist = previousSessionModifiedTime - 2000;
    // If the crash occurred between the last persist and estimated end of session
    return crashTime > lastPersist && crashTime < prevSessionEnd;
}
/** Sets the previous session as the current session and returns any existing session */
async function setPreviousSessionAsCurrent() {
    const previous = await previousSession;
    const scope = core.getIsolationScope();
    const currentSession = scope.getSession();
    if (previous) {
        previousSession = undefined;
        if (previous.status === 'ok') {
            scope.setSession(core.makeSession(previous));
        }
    }
    return currentSession;
}
/** Restores a session */
function restorePreviousSession(session) {
    core.getIsolationScope().setSession(session);
}
/** Report the previous session as abnormal */
async function previousSessionWasAbnormal() {
    const client = core.getClient();
    const previous = await previousSession;
    if (previous && client) {
        // Ignore if the previous session is already ended
        if (previous.status !== 'ok') {
            previousSession = undefined;
            return;
        }
        core.debug.log('Found previous abnormal session');
        const sesh = core.makeSession(previous);
        core.updateSession(sesh, {
            status: 'abnormal',
            errors: (sesh.errors || 0) + 1,
            release: previous.attrs?.release,
            environment: previous.attrs?.environment,
        });
        await client.sendSession(sesh);
        previousSession = undefined;
    }
}
/** Checks if the previous session needs sending as crashed or abnormal  */
async function checkPreviousSession(crashed) {
    const client = core.getClient();
    const previous = await previousSession;
    if (previous && client) {
        // Ignore if the previous session is already ended
        if (previous.status !== 'ok') {
            previousSession = undefined;
            return;
        }
        const status = crashed ? 'crashed' : 'abnormal';
        core.debug.log(`Found previous ${status} session`);
        const sesh = core.makeSession(previous);
        core.updateSession(sesh, {
            status,
            errors: (sesh.errors || 0) + 1,
            release: previous.attrs?.release,
            environment: previous.attrs?.environment,
        });
        await client.sendSession(sesh);
        previousSession = undefined;
    }
}
/** Sets the current session as crashed */
function sessionCrashed() {
    // stop persisting session
    if (persistTimer) {
        clearInterval(persistTimer);
    }
    core.debug.log('Session Crashed');
    const session = core.getIsolationScope().getSession();
    if (!session) {
        core.debug.log('No session to update');
        return;
    }
    if (session.status === 'ok') {
        core.debug.log('Setting session as crashed');
        const errors = session.errors + 1;
        core.updateSession(session, { status: 'crashed', errors });
        core.captureSession();
    }
    else {
        core.debug.log('Session already ended');
    }
}
/** Sets the current session as ANR */
function sessionAnr() {
    // stop persisting session
    if (persistTimer) {
        clearInterval(persistTimer);
    }
    const session = core.getIsolationScope().getSession();
    if (!session) {
        return;
    }
    if (session.status === 'ok') {
        core.debug.log('Setting session as abnormal ANR');
        core.updateSession(session, { status: 'abnormal', abnormal_mechanism: 'anr_foreground' });
        core.captureSession();
    }
}
/**
 * End the current session on app exit
 */
function endSessionOnExit() {
    // 'before-quit' is always called before 'will-quit' so we listen there and ensure our 'will-quit' handler is still
    // the last listener
    electron.app.on('before-quit', () => {
        // We track the end of sessions via the 'will-quit' event which is the last event emitted before close.
        //
        // We need to be the last 'will-quit' listener so as not to interfere with any user defined listeners which may
        // call `event.preventDefault()` to abort the exit.
        electron.app.removeListener('will-quit', exitHandler);
        electron.app.on('will-quit', exitHandler);
    });
}
/** Handles the exit */
const exitHandler = async (event) => {
    if (event.defaultPrevented) {
        return;
    }
    core.debug.log('[Session] Exit Handler');
    // Stop the exit so we have time to send the session
    event.preventDefault();
    try {
        // End the session
        await endSession();
    }
    catch (e) {
        // Ignore and log any errors which would prevent app exit
        core.debug.warn('[Session] Error ending session:', e);
    }
    electron.app.exit();
};

exports.checkPreviousSession = checkPreviousSession;
exports.endSession = endSession;
exports.endSessionOnExit = endSessionOnExit;
exports.previousSessionWasAbnormal = previousSessionWasAbnormal;
exports.restorePreviousSession = restorePreviousSession;
exports.sessionAnr = sessionAnr;
exports.sessionCrashed = sessionCrashed;
exports.setPreviousSessionAsCurrent = setPreviousSessionAsCurrent;
exports.startSession = startSession;
exports.unreportedDuringLastSession = unreportedDuringLastSession;//# sourceMappingURL=http://go/sourcemap/sourcemaps/656af3704923280dedba3ccd49cfaf9b9d456e90/node_modules/@sentry/electron/main/sessions.js.map
