Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const url = require('../../utils/url.js');
const attributes = require('./attributes.js');
const methodConfig = require('./methodConfig.js');
const sessionManagement = require('./sessionManagement.js');

/**
 * Attribute extraction and building functions for MCP server instrumentation
 */


/**
 * Extracts transport types based on transport constructor name
 * @param transport - MCP transport instance
 * @returns Transport type mapping for span attributes
 */
function getTransportTypes(transport) {
  const transportName = transport.constructor?.name?.toLowerCase() || '';

  if (transportName.includes('stdio')) {
    return { mcpTransport: 'stdio', networkTransport: 'pipe' };
  }

  if (transportName.includes('streamablehttp') || transportName.includes('streamable')) {
    return { mcpTransport: 'http', networkTransport: 'tcp' };
  }

  if (transportName.includes('sse')) {
    return { mcpTransport: 'sse', networkTransport: 'tcp' };
  }

  return { mcpTransport: 'unknown', networkTransport: 'unknown' };
}

/**
 * Extracts additional attributes for specific notification types
 * @param method - Notification method name
 * @param params - Notification parameters
 * @returns Method-specific attributes for span instrumentation
 */
function getNotificationAttributes(
  method,
  params,
) {
  const attributes$1 = {};

  switch (method) {
    case 'notifications/cancelled':
      if (params?.requestId) {
        attributes$1['mcp.cancelled.request_id'] = String(params.requestId);
      }
      if (params?.reason) {
        attributes$1['mcp.cancelled.reason'] = String(params.reason);
      }
      break;

    case 'notifications/message':
      if (params?.level) {
        attributes$1[attributes.MCP_LOGGING_LEVEL_ATTRIBUTE] = String(params.level);
      }
      if (params?.logger) {
        attributes$1[attributes.MCP_LOGGING_LOGGER_ATTRIBUTE] = String(params.logger);
      }
      if (params?.data !== undefined) {
        attributes$1[attributes.MCP_LOGGING_DATA_TYPE_ATTRIBUTE] = typeof params.data;
        if (typeof params.data === 'string') {
          attributes$1[attributes.MCP_LOGGING_MESSAGE_ATTRIBUTE] = params.data;
        } else {
          attributes$1[attributes.MCP_LOGGING_MESSAGE_ATTRIBUTE] = JSON.stringify(params.data);
        }
      }
      break;

    case 'notifications/progress':
      if (params?.progressToken) {
        attributes$1['mcp.progress.token'] = String(params.progressToken);
      }
      if (typeof params?.progress === 'number') {
        attributes$1['mcp.progress.current'] = params.progress;
      }
      if (typeof params?.total === 'number') {
        attributes$1['mcp.progress.total'] = params.total;
        if (typeof params?.progress === 'number') {
          attributes$1['mcp.progress.percentage'] = (params.progress / params.total) * 100;
        }
      }
      if (params?.message) {
        attributes$1['mcp.progress.message'] = String(params.message);
      }
      break;

    case 'notifications/resources/updated':
      if (params?.uri) {
        attributes$1[attributes.MCP_RESOURCE_URI_ATTRIBUTE] = String(params.uri);
        const urlObject = url.parseStringToURLObject(String(params.uri));
        if (urlObject && !url.isURLObjectRelative(urlObject)) {
          attributes$1['mcp.resource.protocol'] = urlObject.protocol.replace(':', '');
        }
      }
      break;

    case 'notifications/initialized':
      attributes$1['mcp.lifecycle.phase'] = 'initialization_complete';
      attributes$1['mcp.protocol.ready'] = 1;
      break;
  }

  return attributes$1;
}

/**
 * Extracts and validates PartyInfo from an unknown object
 * @param obj - Unknown object that might contain party info
 * @returns Validated PartyInfo object with only string properties
 */
function extractPartyInfo(obj) {
  const partyInfo = {};

  if (obj && typeof obj === 'object' && obj !== null) {
    const source = obj ;
    if (typeof source.name === 'string') partyInfo.name = source.name;
    if (typeof source.title === 'string') partyInfo.title = source.title;
    if (typeof source.version === 'string') partyInfo.version = source.version;
  }

  return partyInfo;
}

/**
 * Extracts session data from "initialize" requests
 * @param request - JSON-RPC "initialize" request containing client info and protocol version
 * @returns Session data extracted from request parameters including protocol version and client info
 */
function extractSessionDataFromInitializeRequest(request) {
  const sessionData = {};
  if (request.params && typeof request.params === 'object' && request.params !== null) {
    const params = request.params ;
    if (typeof params.protocolVersion === 'string') {
      sessionData.protocolVersion = params.protocolVersion;
    }
    if (params.clientInfo) {
      sessionData.clientInfo = extractPartyInfo(params.clientInfo);
    }
  }
  return sessionData;
}

/**
 * Extracts session data from "initialize" response
 * @param result - "initialize" response result containing server info and protocol version
 * @returns Partial session data extracted from response including protocol version and server info
 */
function extractSessionDataFromInitializeResponse(result) {
  const sessionData = {};
  if (result && typeof result === 'object') {
    const resultObj = result ;
    if (typeof resultObj.protocolVersion === 'string') sessionData.protocolVersion = resultObj.protocolVersion;
    if (resultObj.serverInfo) {
      sessionData.serverInfo = extractPartyInfo(resultObj.serverInfo);
    }
  }
  return sessionData;
}

/**
 * Build client attributes from stored client info
 * @param transport - MCP transport instance
 * @returns Client attributes for span instrumentation
 */
function getClientAttributes(transport) {
  const clientInfo = sessionManagement.getClientInfoForTransport(transport);
  const attributes = {};

  if (clientInfo?.name) {
    attributes['mcp.client.name'] = clientInfo.name;
  }
  if (clientInfo?.title) {
    attributes['mcp.client.title'] = clientInfo.title;
  }
  if (clientInfo?.version) {
    attributes['mcp.client.version'] = clientInfo.version;
  }

  return attributes;
}

/**
 * Build server attributes from stored server info
 * @param transport - MCP transport instance
 * @returns Server attributes for span instrumentation
 */
function getServerAttributes(transport) {
  const serverInfo = sessionManagement.getSessionDataForTransport(transport)?.serverInfo;
  const attributes$1 = {};

  if (serverInfo?.name) {
    attributes$1[attributes.MCP_SERVER_NAME_ATTRIBUTE] = serverInfo.name;
  }
  if (serverInfo?.title) {
    attributes$1[attributes.MCP_SERVER_TITLE_ATTRIBUTE] = serverInfo.title;
  }
  if (serverInfo?.version) {
    attributes$1[attributes.MCP_SERVER_VERSION_ATTRIBUTE] = serverInfo.version;
  }

  return attributes$1;
}

/**
 * Extracts client connection info from extra handler data
 * @param extra - Extra handler data containing connection info
 * @returns Client address and port information
 */
function extractClientInfo(extra)

 {
  return {
    address:
      extra?.requestInfo?.remoteAddress ||
      extra?.clientAddress ||
      extra?.request?.ip ||
      extra?.request?.connection?.remoteAddress,
    port: extra?.requestInfo?.remotePort || extra?.clientPort || extra?.request?.connection?.remotePort,
  };
}

/**
 * Build transport and network attributes
 * @param transport - MCP transport instance
 * @param extra - Optional extra handler data
 * @returns Transport attributes for span instrumentation
 */
function buildTransportAttributes(
  transport,
  extra,
) {
  const sessionId = transport.sessionId;
  const clientInfo = extra ? extractClientInfo(extra) : {};
  const { mcpTransport, networkTransport } = getTransportTypes(transport);
  const clientAttributes = getClientAttributes(transport);
  const serverAttributes = getServerAttributes(transport);
  const protocolVersion = sessionManagement.getProtocolVersionForTransport(transport);

  const attributes$1 = {
    ...(sessionId && { [attributes.MCP_SESSION_ID_ATTRIBUTE]: sessionId }),
    ...(clientInfo.address && { [attributes.CLIENT_ADDRESS_ATTRIBUTE]: clientInfo.address }),
    ...(clientInfo.port && { [attributes.CLIENT_PORT_ATTRIBUTE]: clientInfo.port }),
    [attributes.MCP_TRANSPORT_ATTRIBUTE]: mcpTransport,
    [attributes.NETWORK_TRANSPORT_ATTRIBUTE]: networkTransport,
    [attributes.NETWORK_PROTOCOL_VERSION_ATTRIBUTE]: '2.0',
    ...(protocolVersion && { [attributes.MCP_PROTOCOL_VERSION_ATTRIBUTE]: protocolVersion }),
    ...clientAttributes,
    ...serverAttributes,
  };

  return attributes$1;
}

/**
 * Build type-specific attributes based on message type
 * @param type - Span type (request or notification)
 * @param message - JSON-RPC message
 * @param params - Optional parameters for attribute extraction
 * @returns Type-specific attributes for span instrumentation
 */
function buildTypeSpecificAttributes(
  type,
  message,
  params,
) {
  if (type === 'request') {
    const request = message ;
    const targetInfo = methodConfig.extractTargetInfo(request.method, params || {});

    return {
      ...(request.id !== undefined && { [attributes.MCP_REQUEST_ID_ATTRIBUTE]: String(request.id) }),
      ...targetInfo.attributes,
      ...methodConfig.getRequestArguments(request.method, params || {}),
    };
  }

  return getNotificationAttributes(message.method, params || {});
}

/**
 * Build attributes for tool result content items
 * @param content - Array of content items from tool result
 * @returns Attributes extracted from each content item including type, text, mime type, URI, and resource info
 */
function buildAllContentItemAttributes(content) {
  const attributes$1 = {
    [attributes.MCP_TOOL_RESULT_CONTENT_COUNT_ATTRIBUTE]: content.length,
  };

  for (const [i, item] of content.entries()) {
    if (typeof item !== 'object' || item === null) continue;

    const contentItem = item ;
    const prefix = content.length === 1 ? 'mcp.tool.result' : `mcp.tool.result.${i}`;

    const safeSet = (key, value) => {
      if (typeof value === 'string') attributes$1[`${prefix}.${key}`] = value;
    };

    safeSet('content_type', contentItem.type);
    safeSet('mime_type', contentItem.mimeType);
    safeSet('uri', contentItem.uri);
    safeSet('name', contentItem.name);

    if (typeof contentItem.text === 'string') {
      const text = contentItem.text;
      const maxLength = 500;
      attributes$1[`${prefix}.content`] = text.length > maxLength ? `${text.slice(0, maxLength - 3)}...` : text;
    }

    if (typeof contentItem.data === 'string') {
      attributes$1[`${prefix}.data_size`] = contentItem.data.length;
    }

    const resource = contentItem.resource;
    if (typeof resource === 'object' && resource !== null) {
      const res = resource ;
      safeSet('resource_uri', res.uri);
      safeSet('resource_mime_type', res.mimeType);
    }
  }

  return attributes$1;
}

/**
 * Extract tool result attributes for span instrumentation
 * @param result - Tool execution result
 * @returns Attributes extracted from tool result content
 */
function extractToolResultAttributes(result) {
  let attributes$1 = {};
  if (typeof result !== 'object' || result === null) return attributes$1;

  const resultObj = result ;
  if (typeof resultObj.isError === 'boolean') {
    attributes$1[attributes.MCP_TOOL_RESULT_IS_ERROR_ATTRIBUTE] = resultObj.isError;
  }
  if (Array.isArray(resultObj.content)) {
    attributes$1 = { ...attributes$1, ...buildAllContentItemAttributes(resultObj.content) };
  }
  return attributes$1;
}

exports.buildTransportAttributes = buildTransportAttributes;
exports.buildTypeSpecificAttributes = buildTypeSpecificAttributes;
exports.extractClientInfo = extractClientInfo;
exports.extractSessionDataFromInitializeRequest = extractSessionDataFromInitializeRequest;
exports.extractSessionDataFromInitializeResponse = extractSessionDataFromInitializeResponse;
exports.extractToolResultAttributes = extractToolResultAttributes;
exports.getClientAttributes = getClientAttributes;
exports.getNotificationAttributes = getNotificationAttributes;
exports.getServerAttributes = getServerAttributes;
exports.getTransportTypes = getTransportTypes;//# sourceMappingURL=http://go/sourcemap/sourcemaps/5069385c5a69db511722405ab5aeadc01579afd0/node_modules/@sentry/core/build/cjs/integrations/mcp-server/attributeExtraction.js.map
