import { defineIntegration } from '@sentry/core';
import { eventLoopBlockIntegration as eventLoopBlockIntegration$1 } from '@sentry/node-native';
export { disableBlockDetectionForCallback, pauseEventLoopBlockDetection, restartEventLoopBlockDetection } from '@sentry/node-native';
import { app, powerMonitor } from 'electron';

/**
 * > **Note**
 * >
 * > You should add `@sentry/node-native` to your dependencies to use this integration.
 *
 * Monitors the Node.js event loop for blocking behavior and reports blocked events to Sentry.
 *
 * Uses a background worker and native module to detect when the main thread is blocked for longer than the configured
 * threshold (default: 1 second). When a block is detected, it captures an event with stack traces for every thread.
 *
 * ```js
 * import * as Sentry from '@sentry/electron/main';
 * import { eventLoopBlockIntegration } from '@sentry/electron/native';
 *
 * Sentry.init({
 *   dsn: '__YOUR_DSN__',
 *   integrations: [
 *     eventLoopBlockIntegration({
 *       threshold: 500, // Report blocks longer than 500ms
 *     }),
 *   ],
 * });
 * ```
 */
const eventLoopBlockIntegration = defineIntegration((options = {}) => {
    if (process.type === 'renderer') {
        throw new Error('Detected `eventLoopBlockIntegration` in renderer process. This integration should only be used in the Electron main process.');
    }
    const integration = eventLoopBlockIntegration$1({
        ...options,
        staticTags: {
            'event.environment': 'javascript',
            'event.origin': 'electron',
            'event.process': 'browser',
            ...options.staticTags,
        },
        appRootPath: app.getAppPath(),
    });
    powerMonitor.on('suspend', () => {
        integration.stop();
    });
    powerMonitor.on('lock-screen', () => {
        integration.stop();
    });
    powerMonitor.on('resume', () => {
        integration.start();
    });
    powerMonitor.on('unlock-screen', () => {
        integration.start();
    });
    return integration;
});

export { eventLoopBlockIntegration };//# sourceMappingURL=http://go/sourcemap/sourcemaps/44f52350f2b7b9ee7625cfaade862ff53a016f40/node_modules/@sentry/electron/esm/native/event-loop-blocked-integration.js.map
