"use strict";
// Copyright 2021-2024 The Connect Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.compressionBrotli = exports.compressionGzip = void 0;
const zlib = require("zlib");
const util_1 = require("util");
const connect_1 = require("@connectrpc/connect");
const node_error_js_1 = require("./node-error.js");
const gzip = (0, util_1.promisify)(zlib.gzip);
const gunzip = (0, util_1.promisify)(zlib.gunzip);
const brotliCompress = (0, util_1.promisify)(zlib.brotliCompress);
const brotliDecompress = (0, util_1.promisify)(zlib.brotliDecompress);
/**
 * The gzip compression algorithm, implemented with the Node.js built-in module
 * zlib. This value can be used for the `sendCompression` and `acceptCompression`
 * option of client transports, or for the `acceptCompression` option of server
 * plugins like `fastifyConnectPlugin` from @connectrpc/connect-fastify.
 */
exports.compressionGzip = {
    name: "gzip",
    compress(bytes) {
        return gzip(bytes, {});
    },
    decompress(bytes, readMaxBytes) {
        if (bytes.length == 0) {
            return Promise.resolve(new Uint8Array(0));
        }
        return wrapZLibErrors(gunzip(bytes, {
            maxOutputLength: readMaxBytes,
        }), readMaxBytes);
    },
};
/**
 * The brotli compression algorithm, implemented with the Node.js built-in module
 * zlib. This value can be used for the `sendCompression` and `acceptCompression`
 * option of client transports, or for the `acceptCompression` option of server
 * plugins like `fastifyConnectPlugin` from @connectrpc/connect-fastify.
 */
exports.compressionBrotli = {
    name: "br",
    compress(bytes) {
        return brotliCompress(bytes, {});
    },
    decompress(bytes, readMaxBytes) {
        if (bytes.length == 0) {
            return Promise.resolve(new Uint8Array(0));
        }
        return wrapZLibErrors(brotliDecompress(bytes, {
            maxOutputLength: readMaxBytes,
        }), readMaxBytes);
    },
};
function wrapZLibErrors(promise, readMaxBytes) {
    return promise.catch((e) => {
        const props = (0, node_error_js_1.getNodeErrorProps)(e);
        let code = connect_1.Code.Internal;
        let message = "decompression failed";
        // eslint-disable-next-line @typescript-eslint/switch-exhaustiveness-check
        switch (props.code) {
            case "ERR_BUFFER_TOO_LARGE":
                code = connect_1.Code.ResourceExhausted;
                message = `message is larger than configured readMaxBytes ${readMaxBytes} after decompression`;
                break;
            case "Z_DATA_ERROR":
            case "ERR_PADDING_2":
                code = connect_1.Code.InvalidArgument;
                break;
            default:
                if (props.code !== undefined &&
                    props.code.startsWith("ERR__ERROR_FORMAT_")) {
                    code = connect_1.Code.InvalidArgument;
                }
                break;
        }
        return Promise.reject(new connect_1.ConnectError(message, code, undefined, undefined, e));
    });
}
