import { defineIntegration } from '@sentry/core';
import { app, BrowserWindow } from 'electron';
import { endSessionOnExit, startSession, endSession } from '../sessions.js';

function focusedWindow() {
    for (const window of BrowserWindow.getAllWindows()) {
        if (!window.isDestroyed() && window.webContents && !window.webContents.isDestroyed()) {
            // It's important to test both isVisible and isFocused, since
            // Electron v12-13 do not report hidden as a loss of focus
            if (window.isFocused() && window.isVisible()) {
                return true;
            }
        }
    }
    return false;
}
/**
 * Tracks sessions as BrowserWindows focus.
 *
 * Supports Electron >= v12
 */
const browserWindowSessionIntegration = defineIntegration((options = {}) => {
    let _state = { name: 'inactive' };
    function windowStateChanged() {
        const hasFocusedWindow = focusedWindow();
        if (hasFocusedWindow) {
            // We are now active
            if (_state.name === 'inactive') {
                // If we were inactive, start a new session
                startSession(true);
            }
            else if (_state.name === 'timeout') {
                // Clear the timeout since the app has become active again
                clearTimeout(_state.timer);
            }
            _state = { name: 'active' };
        }
        else {
            if (_state.name === 'active') {
                // We have become inactive, start the timeout
                const timeout = (options.backgroundTimeoutSeconds ?? 30) * 1000;
                const timer = setTimeout(() => {
                    // if the state says we're still waiting for the timeout, end the session
                    if (_state.name === 'timeout') {
                        _state = { name: 'inactive' };
                        endSession().catch(() => {
                            // ignore
                        });
                    }
                }, timeout)
                    // unref so this timer doesn't block app exit
                    .unref();
                _state = { name: 'timeout', timer };
            }
        }
    }
    return {
        name: 'BrowserWindowSession',
        setup() {
            app.on('browser-window-created', (_event, window) => {
                window.on('focus', windowStateChanged);
                window.on('blur', windowStateChanged);
                window.on('show', windowStateChanged);
                window.on('hide', windowStateChanged);
                // when the window is closed we need to remove the listeners
                window.once('closed', () => {
                    window.removeListener('focus', windowStateChanged);
                    window.removeListener('blur', windowStateChanged);
                    window.removeListener('show', windowStateChanged);
                    window.removeListener('hide', windowStateChanged);
                });
            });
            // if the app exits while the session is active, end the session
            endSessionOnExit();
        },
    };
});

export { browserWindowSessionIntegration };//# sourceMappingURL=http://go/sourcemap/sourcemaps/32cfbe848b35d9eb320980195985450f244b3030/node_modules/@sentry/electron/esm/main/integrations/browser-window-session.js.map
