const core = require('@sentry/core');
const node = require('@sentry/node');
const electron = require('electron');
const stackParse = require('../renderer/stack-parse.js');
const electronNormalize = require('./electron-normalize.js');

// node.js stack parser but filename normalized before parsing the module
const defaultStackParser = core.createStackParser(core.nodeStackLineParser(node.createGetModuleFromFilename(electron.app.getAppPath())));
/**
 * Captures stack frames from a renderer process
 *
 * Requires Electron >= 34 and throws an error on older versions
 *
 * @param webContents The WebContents to capture stack frames from
 * @returns A promise that resolves to an array of Sentry StackFrames
 */
async function captureRendererStackFrames(webContents) {
    if (electronNormalize.ELECTRON_MAJOR_VERSION < 34) {
        throw new Error('Electron >= 34 required to capture stack frames via `frame.collectJavaScriptCallStack()`');
    }
    if (webContents.isDestroyed()) {
        return undefined;
    }
    const frame = webContents.mainFrame;
    const stack = await frame.collectJavaScriptCallStack();
    if (!stack) {
        return undefined;
    }
    if (stack.includes('Website owner has not opted in')) {
        core.debug.warn("Could not collect renderer stack frames.\nA 'Document-Policy' header of 'include-js-call-stacks-in-crash-reports' must be set");
        return undefined;
    }
    return stackParse.electronRendererStackParser(stack);
}

exports.captureRendererStackFrames = captureRendererStackFrames;
exports.defaultStackParser = defaultStackParser;//# sourceMappingURL=http://go/sourcemap/sourcemaps/2e353c5f5b30150ff7b874dee5a87660693d9de0/node_modules/@sentry/electron/main/stack-parse.js.map
